/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.calendar.impl;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.streams.ReadStream;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.calendar.api.IVEvent;
import net.bluemind.calendar.api.internal.CalendarHistory;
import net.bluemind.calendar.api.internal.IInternalCalendar;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.dataprotect.api.IBackupWorker;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.calendar.impl.CalendarBackupRepository.BackupResourceCalendarRecord;
import net.bluemind.dataprotect.calendar.impl.CalendarBackupRepository.BackupUserCalendarRecord;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.dataprotect.service.tool.CoreCommonBackupWorker;
import net.bluemind.dataprotect.service.tool.ZipBuilder;
import net.bluemind.directory.api.BaseDirEntry;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.lib.vertx.utils.MmapWriteStream;
import net.bluemind.resource.api.IResources;
import net.bluemind.resource.api.ResourceDescriptor;
import net.bluemind.server.api.Server;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public class CalendarBackupWorker implements IBackupWorker {
	private static Logger logger = LoggerFactory.getLogger(CalendarBackupWorker.class);
	public static final Path WORKING_PATH = ZipBuilder.WORKING_PATH.resolve("calendars");
	private static final String ZIP_FILE_PREFIX = ICalendarUids.TYPE;

	private static class CalendarBackup extends CommonBackupWorker {
		private CalendarBackupRepository repository;

		public CalendarBackup(IDPContext ctx, Path p, PartGeneration partGen) {
			super(ctx, partGen);
			repository = new CalendarBackupRepository(p);
		}

		@Override
		public List<MailboxIndexJson> partitionTreatment(String domainUid, String zipFile, List<String> partition) {
			IDirectory dirApi = provider().instance(IDirectory.class, domainUid);
			String zipFileName = CommonBackupWorker.zipFilename(ZIP_FILE_PREFIX, zipFile);
			List<MailboxIndexJson> indexJson = new ArrayList<>();
			partition.forEach(uid -> {
				try {
					DirEntry dirEntry = dirApi.findByEntryUid(uid);
					if (dirEntry.kind == Kind.USER || dirEntry.kind == Kind.SHARED_MAILBOX) {
						List<String> files = repository.writeUsers(backupCalendarUser(uid, dirEntry.kind));
						indexJson.add(new MailboxIndexJson(files, uid, domainUid, zipFileName));
					} else if (dirEntry.kind == Kind.RESOURCE) {
						List<String> files = repository.writeResources(backupCalendarResource(uid));
						indexJson.add(new MailboxIndexJson(files, uid, domainUid, zipFileName));
					}
				} catch (IOException e) {
					logError(e, "Unable to backup calendars {}", uid);
				}
			});
			indexJson.forEach(i -> i.zipFileName(zipFileName));
			indexJson.forEach(i -> i.domainUid(domainUid));
			return indexJson;
		}

		private MailboxIndexJsonList backupUsersCalendars(ItemValue<Domain> domain) {
			var q = DirEntryQuery.allWithHidden();
			q.kindsFilter = Arrays.asList(BaseDirEntry.Kind.USER, BaseDirEntry.Kind.RESOURCE,
					BaseDirEntry.Kind.SHARED_MAILBOX);
			ListResult<String> searchUids = provider().instance(IDirectory.class, domain.uid).searchUids(q);
			return runIt(domain.uid, searchUids.values);
		}

		private List<BackupUserCalendarRecord> backupCalendarUser(String entryUid, Kind kind) {
			List<BackupUserCalendarRecord> usersToBackup = new ArrayList<>();
			try {
				provider().instance(IContainers.class) //
						.all(ContainerQuery.ownerAndType(entryUid, ICalendarUids.TYPE)) //
						.forEach(cd -> usersToBackup.add(backupUserCalendar(cd, kind)));
			} catch (Exception e) {
				logError(e, "Error trying to backup {} calendars for {}", kind, entryUid);
			}

			return usersToBackup;
		}

		private List<BackupResourceCalendarRecord> backupCalendarResource(String entryUid) {
			List<BackupResourceCalendarRecord> resourcesToBackup = new ArrayList<>();
			try {
				provider().instance(IContainers.class) //
						.all(ContainerQuery.ownerAndType(entryUid, ICalendarUids.TYPE)) //
						.forEach(cd -> resourcesToBackup.add(backupResourceCalendar(cd)));
			} catch (Exception e) {
				logError(e, "Error trying to backup resource calendars for {}", entryUid);
			}

			return resourcesToBackup;
		}

		private void backupDomainCalendar(String domainUid) {
			try {
				var q = DirEntryQuery.allWithHidden();
				q.kindsFilter = Arrays.asList(BaseDirEntry.Kind.CALENDAR);
				provider().instance(IDirectory.class, domainUid).searchUids(q).values
						.forEach(entryUid -> provider().instance(IContainers.class) //
								.all(ContainerQuery.ownerAndType(entryUid, ICalendarUids.TYPE)) //
								.forEach(containerDescriptor -> {
									try {
										logger.info("Backup domain calendar {}", containerDescriptor.uid);
										IVEvent veventApi = provider().instance(IVEvent.class, containerDescriptor.uid);
										IDomains domainApi = provider().instance(IDomains.class);
										ItemValue<Domain> domain = domainApi.get(containerDescriptor.domainUid);
										IInternalCalendar internalCalendarApi = provider()
												.instance(IInternalCalendar.class, containerDescriptor.uid);
										CalendarHistory calendarHistory = internalCalendarApi.calendarHistory();
										repository.writeDomainCalendar(containerDescriptor, domain,
												readCalendarIcs(containerDescriptor, veventApi), calendarHistory);
									} catch (IOException e) {
										logError(e, "Unable to backup calendar {}", containerDescriptor);
									}
								}));
			} catch (Exception e) {
				logError(e, "Error trying to backup calendars for {}", domainUid);
			}

		}

		private BackupUserCalendarRecord backupUserCalendar(ContainerDescriptor containerDescriptor, Kind kind) {
			logger.info("Backup {} calendar {}", kind, containerDescriptor.uid);
			IVEvent veventApi = provider().instance(IVEvent.class, containerDescriptor.uid);
			IUser userApi = provider().instance(IUser.class, containerDescriptor.domainUid);
			ItemValue<User> user = userApi.getComplete(containerDescriptor.owner);
			IInternalCalendar internalCalendarApi = provider().instance(IInternalCalendar.class,
					containerDescriptor.uid);
			CalendarHistory calendarHistory = internalCalendarApi.calendarHistory();
			ByteBuf out = readCalendarIcs(containerDescriptor, veventApi);
			return new BackupUserCalendarRecord(containerDescriptor, user, out, calendarHistory);
		}

		private BackupResourceCalendarRecord backupResourceCalendar(ContainerDescriptor containerDescriptor) {
			logger.info("Backup resource calendar {}", containerDescriptor.uid);
			IVEvent veventApi = provider().instance(IVEvent.class, containerDescriptor.uid);
			IResources resourceApi = provider().instance(IResources.class, containerDescriptor.domainUid);
			ItemValue<ResourceDescriptor> resource = resourceApi.getComplete(containerDescriptor.owner);
			IInternalCalendar internalCalendarApi = provider().instance(IInternalCalendar.class,
					containerDescriptor.uid);
			CalendarHistory calendarHistory = internalCalendarApi.calendarHistory();
			ByteBuf out = readCalendarIcs(containerDescriptor, veventApi);
			return new BackupResourceCalendarRecord(containerDescriptor, resource, out, calendarHistory);
		}

		private ByteBuf readCalendarIcs(ContainerDescriptor containerDescriptor, IVEvent veventApi) {
			try {
				MmapWriteStream target = new MmapWriteStream(CalendarBackupWorker.WORKING_PATH, Integer.MAX_VALUE);
				Stream ics = veventApi.exportAll();
				ReadStream<Buffer> icsStream = VertxStream.read(ics);
				VertxPlatform.getVertx().executeBlocking(() -> icsStream.pipeTo(target), false);
				return target.mmap().orTimeout(5, TimeUnit.MINUTES).join();
			} catch (Exception e) {
				logger.warn("Unable to backup ics data for calendar {}: {}", containerDescriptor.uid, e.getMessage());
				return Unpooled.EMPTY_BUFFER;
			}
		}

		public void backupAll() {
			var domainsApi = provider().instance(IInCoreDomains.class);
			List<MailboxIndexJson> domainsZip = new ArrayList<>();
			domainsApi.allUnfiltered().stream().filter(d -> !"global.virt".equals(d.uid)).forEach(domain -> {
				logger.info("Backup calendars for domain {}", domain.value.defaultAlias);
				backupDomainCalendar(domain.uid);
				domainsZip.addAll(backupUsersCalendars(domain).mailboxesIndexInfo());
			});

			Path rootUserPath = repository.repositoryPath.resolveRootUserPath();
			try {
				RepositoryBackupPath.writeIndexFileToDir(rootUserPath, domainsZip);
			} catch (IOException e) {
				logError(e, "Error trying to write index.json {}", rootUserPath);
			}
		}
	}

	@Override
	public void prepareDataDirs(IDPContext ctx, PartGeneration partGen, ItemValue<Server> toBackup) throws ServerFault {
		cleanup(ctx, partGen, null);
		Path outputFolder = CalendarBackupRepository.DEFAULT_PATH;
		long startDate = Instant.now().getEpochSecond();

		try {
			if (!Files.exists(WORKING_PATH)) {
				Files.createDirectories(WORKING_PATH);
			}

			logger.info("Starting calendars backup in {}", WORKING_PATH);
			ctx.info("Starting calendars backup");
			new CalendarBackup(ctx, WORKING_PATH, partGen).backupAll();
			new CoreCommonBackupWorker(ctx, toBackup, WORKING_PATH, outputFolder, partGen).prepareDataDirs();
		} catch (IOException e) {
			partGen.withErrors = true;
			ctx.error(e, "Unable to create temporary directory for calendar backup");
		} finally {
			ctx.info(
					String.format("Ending calendars backup in %d seconds", Instant.now().getEpochSecond() - startDate));
		}
	}

	@Override
	public String getDataType() {
		return WorkerDataType.CALENDAR.value;
	}

	@Override
	public Set<String> getDataDirs() {
		return Set.of(CalendarBackupRepository.DEFAULT_PATH.toString());
	}

	@Override
	public boolean supportsTag(String tag) {
		return CoreCommonBackupWorker.supportsTag(tag);
	}
}
