/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2016
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.dataprotect.calendar.impl.user;

import java.util.HashMap;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.CalendarDescriptor;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.calendar.api.internal.CalendarHistory;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.calendar.CalendarRestorableBackupItem;
import net.bluemind.dataprotect.calendar.impl.AbstractRestoreCalendar;
import net.bluemind.dataprotect.calendar.impl.CalendarBackupRepository;
import net.bluemind.dataprotect.calendar.impl.CommonRestoreCalendar;
import net.bluemind.dataprotect.calendar.impl.pg.RestoreUserCalendarsTaskPg;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;

public class RestoreUserCalendarsTask extends AbstractRestoreCalendar implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(RestoreUserCalendarsTask.class);

	protected final RestoreUserCalendarsWorker userCalendarsWorker;

	public RestoreUserCalendarsTask(DataProtectGeneration backup, Restorable item) {
		super(backup, item);
		this.userCalendarsWorker = new RestoreUserCalendarsWorker(backup, restorableItem);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startExportMsg(DPError.DPKind.CALENDAR.name()));
		if (logger.isInfoEnabled()) {
			logger.info("Starting restore for {}@{}", restorableItem.entryUid(), restorableItem.domain());
		}

		CalendarBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreUserCalendarsTaskPg(backup, restorableItem).run();
		}

	}

	protected void restore(CalendarBackupRepository repository) {
		try {
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
			List<String> calendarUids = repository.getRestorableCalendarUids(restorableItem);
			CommonRestoreCalendar commonRestoreCalendar = new CommonRestoreCalendar(restorableItem, live);
			for (String calendarUid : calendarUids) {
				try (CalendarRestorableBackupItem restorable = repository.getRestorableDirCalendar(restorableItem,
						calendarUid)) {

					ContainerDescriptor cd = restorable.backupDescriptor.getContainer();
					CalendarDescriptor calendarDescriptor = CalendarDescriptor.create(cd.name, cd.owner, cd.domainUid);
					if (cd.settings != null) {
						calendarDescriptor.settings = new HashMap<>();
						calendarDescriptor.settings.putAll(cd.settings);
					}

					restorableItem.monitorLog("Restore calendar {}@{} ({})", calendarDescriptor.name,
							calendarDescriptor.domainUid, calendarUid);

					userCalendarsWorker.createOrUpdateLiveEntry(ItemValue.create(calendarUid, calendarDescriptor));

					List<ItemValue<VEventSeries>> events = importEvents(restorable, cd.owner, cd.domainUid,
							restorableItem.item().kind);

					restoreCalendarEntities(restorable, commonRestoreCalendar, calendarUid, events);
				}

			}
		} catch (Exception e) {
			logger.error("Error while restoring calendar {}@{}: {}", restorableItem.entryUid(), restorableItem.domain(),
					e.getMessage());
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.CALENDAR));

		} finally {
			restorableItem.endTask();
		}
	}

	private void restoreCalendarEntities(CalendarRestorableBackupItem restorableCalendar,
			CommonRestoreCalendar commonRestoreCalendar, String calendarUid, List<ItemValue<VEventSeries>> events) {
		try {
			if (restorableCalendar.calendarHistory != null) {
				CalendarHistory calendarHistory = (CalendarHistory) restorableCalendar.calendarHistory.backupDescriptor
						.getItem();
				commonRestoreCalendar.restoreEntitiesWithHistory(events, calendarHistory, calendarUid);
			} else {
				commonRestoreCalendar.restoreEntities(events, calendarUid);
			}
		} catch (Exception e) {
			logger.error("Error while restoring calendar entities with history {}@{}: {}", restorableItem.entryUid(),
					restorableItem.domain(), e.getMessage());
			logger.info("Try to restore calendar entities without history {}@{}: {}", restorableItem.entryUid(),
					restorableItem.domain(), e.getMessage());
			commonRestoreCalendar.restoreEntities(events, calendarUid);
		}
	}
}
