/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.calendar.impl.pg;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.CalendarDescriptor;
import net.bluemind.calendar.api.ICalendarsMgmt;
import net.bluemind.calendar.api.IVEvent;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.core.sessions.Sessions;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.email.SendIcs;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;

public class SendDomainCalendarsICSTaskPg {
	private static final Logger logger = LoggerFactory.getLogger(SendDomainCalendarsICSTaskPg.class);

	private final DataProtectGeneration backup;
	private IMonitoredRestoreRestorableItem restorableItem;
	private final RestoreActionExecutor<EmailData> executor;
	private final ResourceBundle bundle;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@Deprecated
	@SuppressWarnings("unchecked")
	public SendDomainCalendarsICSTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem item,
			RestoreActionExecutor<? extends IRestoreActionData> executor, ResourceBundle bundle) {
		this.backup = backup;
		this.restorableItem = item;
		this.bundle = bundle;
		this.executor = (RestoreActionExecutor<EmailData>) executor;
	}

	public static final SecurityContext as(String uid, String domainContainerUid) throws ServerFault {
		SecurityContext userContext = new SecurityContext(UUID.randomUUID().toString(), uid, Arrays.<String>asList(),
				Arrays.<String>asList(), Collections.emptyMap(), domainContainerUid, "en",
				"SendUserCalendarsICSTask.as");
		Sessions.get().put(userContext.getSessionId(), userContext);
		return userContext;
	}

	public void run() throws Exception {
		if (logger.isInfoEnabled()) {
			logger.info("Starting restore for uid {}", restorableItem.entryUid());
		}
		SendIcs sendMail = new SendIcs(restorableItem, executor);

		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext back = bdp.createContextWithData(backup, restorableItem.item());

			CalendarDescriptor backDomainCalendar = back.provider().instance(ICalendarsMgmt.class)
					.get(restorableItem.entryUid());

			IVEvent service = back.provider().instance(IVEvent.class, restorableItem.entryUid());
			Map<String, String> allIcs = Map.of(backDomainCalendar.name,
					GenericStream.streamToString(service.exportAll()));

			restorableItem.monitorLog("Export '{}' events from {} [uid={}]", allIcs.size(), backDomainCalendar.name,
					restorableItem.entryUid());

			sendMail.sendMessage(allIcs, bundle.getString("send.calendar.restore.message"),
					bundle.getString("send.calendar.restore.subject"));

		} catch (Exception e) {
			logger.error("Error while sending domain calendars", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.DOMAIN_CALENDAR));
		} finally {
			restorableItem.endTask();
		}
	}

}
