/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2023
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.dataprotect.calendar.impl.domain;

import java.util.HashMap;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.CalendarDescriptor;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.api.RestorableKind;
import net.bluemind.dataprotect.calendar.impl.AbstractRestoreCalendar;
import net.bluemind.dataprotect.calendar.impl.CalendarBackupRepository;
import net.bluemind.dataprotect.calendar.impl.CommonRestoreCalendar;
import net.bluemind.dataprotect.calendar.impl.pg.RestoreDomainCalendarsTaskPg;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.domain.api.Domain;

public class RestoreDomainCalendarsTask extends AbstractRestoreCalendar implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(RestoreDomainCalendarsTask.class);

	protected final RestoreDomainCalendarsWorker domainCalendarsWorker;

	public RestoreDomainCalendarsTask(DataProtectGeneration backup, Restorable item) {
		super(backup, item);
		this.domainCalendarsWorker = new RestoreDomainCalendarsWorker(backup, restorableItem);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPError.DPKind.DOMAIN_CALENDAR.name()));
		if (logger.isInfoEnabled()) {
			logger.info("Starting restore for uid {}", restorableItem.entryUid());
		}

		CalendarBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreDomainCalendarsTaskPg(backup, restorableItem, domainCalendarsWorker).run();
		}

	}

	protected void restore(CalendarBackupRepository repository) {
		try {
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
			try (RestorableContainerItemBackupItem<Domain> restorable = repository
					.getRestorableDomainCalendar(restorableItem.domain(), restorableItem.entryUid())) {
				ContainerDescriptor cd = restorable.backupDescriptor.getContainer();
				CalendarDescriptor calendar = CalendarDescriptor.create(cd.name, cd.owner, cd.domainUid);
				if (cd.settings != null) {
					calendar.settings = new HashMap<>();
					calendar.settings.putAll(cd.settings);
				}

				List<ItemValue<VEventSeries>> events = importEvents(restorable, cd.owner, cd.domainUid,
						RestorableKind.CALENDAR);
				restorableItem.monitorLog("Restore domain calendar {}@{} ({})", calendar.name, calendar.domainUid,
						cd.uid);

				domainCalendarsWorker.createOrUpdateLiveEntry(ItemValue.create(restorableItem.entryUid(), calendar));

				new CommonRestoreCalendar(restorableItem, live).restoreEntities(events, restorableItem.liveEntryUid());
			}
		} catch (Exception e) {
			logger.error("Error while restoring domain calendar {}@{}: {}", restorableItem.entryUid(),
					restorableItem.domain(), e.getMessage());
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.DOMAIN_CALENDAR));
		} finally {
			restorableItem.endTask();
		}
	}

}
