/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.calendar.impl;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.calendar.api.ICalendar;
import net.bluemind.calendar.api.VEventChanges;
import net.bluemind.calendar.api.VEventChanges.ItemAdd;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.calendar.api.internal.CalendarHistory;
import net.bluemind.calendar.api.internal.IInternalCalendar;
import net.bluemind.core.container.model.ContainerUpdatesResult;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.common.restore.CommonRestoreEntitiesFromDatabase;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;

public class CommonRestoreCalendar extends CommonRestoreEntitiesFromDatabase<VEventSeries> {
	private static final Logger logger = LoggerFactory.getLogger(CommonRestoreCalendar.class);

	public CommonRestoreCalendar(IMonitoredRestoreRestorableItem item, BmContext back, BmContext live) {
		super(item, back, live);
	}

	public CommonRestoreCalendar(IMonitoredRestoreRestorableItem item, BmContext live) {
		super(item, live);
	}

	@Override
	public void restoreEntities(List<String> allUids) {
		restoreEntities(allUids, item.entryUid(), item.liveEntryUid());
	}

	@Override
	public void restoreEntities(List<String> allUids, String backUid, String liveUid) {
		ICalendar backCalApi = back.provider().instance(ICalendar.class, backUid);

		for (List<String> batch : Lists.partition(backCalApi.all(), 1000)) {
			List<ItemValue<VEventSeries>> events = backCalApi.multipleGet(batch);
			restoreEntities(events, liveUid);
		}
	}

	@Override
	public void restoreEntities(List<ItemValue<VEventSeries>> backVEventSeries, String liveUid) {
		VEventChanges changes = getEventsToRestore(backVEventSeries, liveUid);

		ICalendar liveCalApi = live.provider().instance(ICalendar.class, liveUid);
		ContainerUpdatesResult updatesResult = liveCalApi.updates(changes);
		if (updatesResult.errors != null && !updatesResult.errors.isEmpty()) {
			item.errors().addAll(updatesResult.errors.stream().map(DPError::createRestore).toList());
		}
		item.monitorLog("{} events to restore ", backVEventSeries.size());
	}

	public void restoreEntitiesWithHistory(List<ItemValue<VEventSeries>> backVEventSeries,
			CalendarHistory calendarHistory, String liveUid) {
		VEventChanges changes = getEventsToRestore(backVEventSeries, liveUid);

		IInternalCalendar liveCalApi = live.provider().instance(IInternalCalendar.class, liveUid);
		ContainerUpdatesResult updatesResult = liveCalApi.restoreEvents(changes, calendarHistory);
		if (updatesResult.errors != null && !updatesResult.errors.isEmpty()) {
			item.errors().addAll(updatesResult.errors.stream().map(DPError::createRestore).toList());
		}
		item.monitorLog("{} events to restore with history", backVEventSeries.size());
	}

	private VEventChanges getEventsToRestore(List<ItemValue<VEventSeries>> backVEventSeries, String liveUid) {
		List<ItemValue<VEventSeries>> eventsToRestore = backVEventSeries.stream().map(be -> {
			List<ItemValue<VEventSeries>> byIcsUid = live.provider().instance(ICalendar.class, liveUid)
					.getByIcsUid(be.value.icsUid);
			if (!byIcsUid.isEmpty()) {
				be.uid = byIcsUid.get(0).uid;
			}
			return be;
		}).toList();

		return VEventChanges.create(
				eventsToRestore.stream().map(e -> ItemAdd.create(e.uid, e.value, false)).collect(Collectors.toList()),
				Collections.emptyList(), Collections.emptyList());
	}
}
