/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.calendar.impl.pg;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.ICalendar;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.calendar.impl.CommonRestoreCalendar;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;

public class RestoreUserCalendarsTaskPg {

	private static final Logger logger = LoggerFactory.getLogger(RestoreUserCalendarsTaskPg.class);
	private IMonitoredRestoreRestorableItem restorableItem;
	private DataProtectGeneration backup;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@Deprecated
	public RestoreUserCalendarsTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem restorableItem) {
		this.backup = backup;
		this.restorableItem = restorableItem;
	}

	public void run() {
		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext back = bdp.createContextWithData(backup, restorableItem.item());
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();

			List<ContainerDescriptor> backCalendars = back.provider().instance(IContainers.class)
					.all(ContainerQuery.ownerAndType(restorableItem.entryUid(), ICalendarUids.TYPE));

			restorableItem.monitorLog("Restoring '{}' calendar(s) from uid={}", backCalendars.size(),
					restorableItem.entryUid());
			logger.info("Backup contains {} calendar(s)", backCalendars.size());

			CommonRestoreCalendar restoreCal = new CommonRestoreCalendar(restorableItem, back, live);
			for (ContainerDescriptor backCalendar : backCalendars) {
				restorableItem.monitorSubWork(1);
				restore(restoreCal, backCalendar);
			}
		} catch (Exception e) {
			logger.error("Error while restoring calendars", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.CALENDAR));
		} finally {
			restorableItem.endTask();
		}
	}

	private void restore(CommonRestoreCalendar restoreCal, ContainerDescriptor backCalendar) {
		ICalendar backCalApi = restoreCal.back.provider().instance(ICalendar.class, backCalendar.uid);

		List<String> allUids = backCalApi.all();
		restorableItem.monitorLog("Restoring {} [uid={}]", backCalendar.name, backCalendar.uid);

		String calendarUid = mapCalendarUid(backCalendar.uid);
		reset(restoreCal, backCalendar, calendarUid);
		restoreCal.restoreEntities(allUids, backCalendar.uid, calendarUid);
	}

	private String mapCalendarUid(String uid) {
		if (!restorableItem.entryUid().equals(restorableItem.liveEntryUid())
				&& uid.endsWith(String.format(":%s", restorableItem.entryUid()))) {
			return String.format("%s%s", uid.substring(0, uid.length() - restorableItem.entryUid().length()),
					restorableItem.liveEntryUid());
		}

		return uid;
	}

	private void reset(CommonRestoreCalendar restoreCal, ContainerDescriptor backCalendar, String calendarUid) {
		List<ContainerDescriptor> liveCalendars = restoreCal.live.provider().instance(IContainers.class)
				.all(ContainerQuery.ownerAndType(restorableItem.liveEntryUid(), ICalendarUids.TYPE));

		if (liveCalendars.stream().noneMatch(c -> c.uid.equals(calendarUid))) {
			backCalendar.owner = restorableItem.liveEntryUid();
			restoreCal.live.provider().instance(IContainers.class).create(calendarUid, backCalendar);
			restorableItem.monitorLog("calendar recreated");
		}
	}
}
