/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.calendar.impl;

import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import net.bluemind.calendar.api.internal.CalendarHistory;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.calendar.CalendarRestorableBackupItem;
import net.bluemind.dataprotect.common.backup.BackupContainerItemDescriptor;
import net.bluemind.dataprotect.common.backup.BackupItemDescriptor;
import net.bluemind.dataprotect.common.backup.JsonReadWrite;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.common.backup.RestorableItemBackupItem;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.domain.api.Domain;
import net.bluemind.resource.api.ResourceDescriptor;
import net.bluemind.user.api.User;

public class CalendarBackupRepository {
	public static final Path DEFAULT_PATH = Paths.get("/var/backups/bluemind/calendars");
	private static Logger logger = LoggerFactory.getLogger(CalendarBackupRepository.class);
	RepositoryBackupPath repositoryPath;

	public CalendarBackupRepository(Path repository) {
		this.repositoryPath = new RepositoryBackupPath(repository);
	}

	record BackupUserCalendarRecord(ContainerDescriptor cd, ItemValue<User> user, ByteBuf ics,
			CalendarHistory calendarHistory) {
	}

	record BackupResourceCalendarRecord(ContainerDescriptor cd, ItemValue<ResourceDescriptor> resource, ByteBuf ics,
			CalendarHistory calendarHistory) {
	}

	private String writeUserCalendarDescriptor(BackupUserCalendarRecord calRecord) throws IOException {
		Path jsonPathToWrite = repositoryPath.prepareJsonPathToWrite(calRecord.cd);
		JsonReadWrite.writeUser(jsonPathToWrite, new BackupContainerItemDescriptor<User>(calRecord.cd, calRecord.user));
		return RepositoryBackupPath.getFilePathWithDirectParent(jsonPathToWrite);
	}

	private String writeResourceCalendarDescriptor(BackupResourceCalendarRecord calRecord) throws IOException {
		Path jsonPathToWrite = repositoryPath.prepareJsonPathToWrite(calRecord.cd);
		JsonReadWrite.writeResource(jsonPathToWrite,
				new BackupContainerItemDescriptor<ResourceDescriptor>(calRecord.cd, calRecord.resource));
		return RepositoryBackupPath.getFilePathWithDirectParent(jsonPathToWrite);
	}

	public List<String> writeUsers(List<BackupUserCalendarRecord> calsToBackup) throws IOException {
		List<String> ret = new ArrayList<>();
		for (BackupUserCalendarRecord entryJson : calsToBackup) {
			ret.add(writeUserCalendarDescriptor(entryJson));
			ret.add(writeBuffer(entryJson.cd, entryJson.ics));
			ret.add(writeHistory(entryJson.cd, entryJson.calendarHistory));
		}
		return ret;
	}

	public List<String> writeResources(List<BackupResourceCalendarRecord> calsToBackup) throws IOException {
		List<String> ret = new ArrayList<>();
		for (BackupResourceCalendarRecord entryJson : calsToBackup) {
			ret.add(writeResourceCalendarDescriptor(entryJson));
			ret.add(writeBuffer(entryJson.cd, entryJson.ics));
			ret.add(writeHistory(entryJson.cd, entryJson.calendarHistory));
		}
		return ret;
	}

	private void writeDomainCalendarDescriptor(ContainerDescriptor containerDescriptor, ItemValue<Domain> domain)
			throws IOException {
		JsonReadWrite.writeDomain(repositoryPath.prepareJsonPathToWrite(containerDescriptor),
				new BackupContainerItemDescriptor<Domain>(containerDescriptor, domain));
	}

	private String writeBuffer(ContainerDescriptor containerDescriptor, ByteBuf content) throws IOException {
		Path target = repositoryPath.getStoragePath(containerDescriptor, RepositoryBackupPath.ICS_FILE_EXTENSION);
		try (SeekableByteChannel chan = Files.newByteChannel(target, StandardOpenOption.CREATE,
				StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING)) {
			chan.write(content.nioBuffer());
		}
		return RepositoryBackupPath.getFilePathWithDirectParent(target);
	}

	private String writeHistory(ContainerDescriptor containerDescriptor, CalendarHistory calendarHistory)
			throws IOException {
		Path pathToWrite = repositoryPath.preparePathToWrite(containerDescriptor,
				RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION);
		JsonReadWrite.writeHistory(pathToWrite,
				new BackupItemDescriptor<CalendarHistory>(containerDescriptor.uid, calendarHistory));
		return RepositoryBackupPath.getFilePathWithDirectParent(pathToWrite);
	}

	public void writeDomainCalendar(ContainerDescriptor containerDescriptor, ItemValue<Domain> domain, ByteBuf ics,
			CalendarHistory calendarHistory) throws IOException {
		writeDomainCalendarDescriptor(containerDescriptor, domain);
		writeBuffer(containerDescriptor, ics);
		writeHistory(containerDescriptor, calendarHistory);
	}

	/**
	 * Returns the list of calendars uids available in the backup
	 *
	 * @param ownerUid
	 * @return List of uid of calendars
	 * @throws IOException
	 */
	public List<String> getRestorableCalendarUids(IMonitoredRestoreRestorableItem item) throws IOException {
		switch (item.item().kind) {
		case USER, RESOURCE: {
			return getRestorableUserCalendarUids(item.domain(), item.entryUid());
		}
		default:
			throw new IllegalArgumentException("Unexpected value: " + item.item().kind);
		}
	}

	private List<String> getRestorableUserCalendarUids(String domainUid, String ownerUid) throws IOException {
		return repositoryPath.getRestorableUserContainerUids(domainUid, ownerUid);
	}

	private List<String> getRestorableResourceCalendarUids(String domainUid, String ownerUid) throws IOException {
		List<String> calDescriptorsUids = Collections.emptyList();
		try {
			calDescriptorsUids = readZipFilesForResourceCalUids(domainUid, ownerUid);
		} catch (Exception e) {
			logger.warn(e.getMessage());
		}

		if (calDescriptorsUids.isEmpty()) {
			calDescriptorsUids = readJsonFilesForResourceCalUids(ownerUid);
		}

		return calDescriptorsUids;
	}

	private List<String> readZipFilesForResourceCalUids(String domainUid, String ownerUid) throws Exception {
		Optional<Path> zipFilePathForOwner = repositoryPath.getZipFilePathForOwner(domainUid, ownerUid);
		if (zipFilePathForOwner.isEmpty()) {
			return Collections.emptyList();
		}

		List<BackupContainerItemDescriptor<ResourceDescriptor>> calDescriptorsFromZip = extractJsonResourceContainerFromZip(
				ownerUid, zipFilePathForOwner.get());

		if (!calDescriptorsFromZip.isEmpty()) {
			return calDescriptorsFromZip.stream().map(ab -> ab.getContainer().uid).toList();
		}

		return Collections.emptyList();
	}

	private List<String> readJsonFilesForResourceCalUids(String ownerUid) throws IOException {
		Path restorablesPath = repositoryPath.resolveUserPath(ownerUid);
		List<BackupContainerItemDescriptor<ResourceDescriptor>> calendars = (List<BackupContainerItemDescriptor<ResourceDescriptor>>) JsonReadWrite
				.readJsonFile(JsonReadWrite.readerResource(), restorablesPath);
		return calendars.stream().map(ab -> ab.getContainer().uid).distinct().toList();
	}

	private List<BackupContainerItemDescriptor<ResourceDescriptor>> extractJsonResourceContainerFromZip(
			String folderName, Path zipFilePath) {
		List<BackupContainerItemDescriptor<ResourceDescriptor>> userContainers = new ArrayList<>();
		try {
			JsonReadWrite.findJsonFilesWithPattern(folderName, zipFilePath).forEach(byteArray -> {
				try {
					userContainers.add((BackupContainerItemDescriptor<ResourceDescriptor>) JsonReadWrite
							.readByteArr(JsonReadWrite.readerResource(), byteArray));
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			});
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return userContainers;
	}

	public CalendarRestorableBackupItem getRestorableDirCalendar(IMonitoredRestoreRestorableItem item,
			String calendarUid) throws Exception {
		switch (item.item().kind) {
		case USER: {
			return getRestorableUserCalendar(item.domain(), item.entryUid(), calendarUid);
		}
		case RESOURCE: {
			return getRestorableResourceCalendar(item.domain(), item.entryUid(), calendarUid);
		}
		default:
			throw new IllegalArgumentException("Unexpected value: " + item.item().kind);
		}
	}

	private Optional<CalendarRestorableBackupItem<User>> readZipFilesForUserCal(String domainUid, String ownerUid,
			String calendarUid) throws Exception {
		return repositoryPath.getZipFilePathForOwner(domainUid, ownerUid).map(zipFilePath -> {
			return extractUserCalendarDataFromZip(ownerUid, calendarUid, zipFilePath);
		});
	}

	private CalendarRestorableBackupItem<User> readJsonFilesForUserCal(String ownerUid, String calendarUid)
			throws IOException {
		Path jsonPath = repositoryPath.resolveUserPathForContainer(ownerUid, calendarUid,
				RepositoryBackupPath.JSON_FILE_EXTENSION);
		Path icsPath = repositoryPath.resolveUserPathForContainer(ownerUid, calendarUid,
				RepositoryBackupPath.ICS_FILE_EXTENSION);
		Path histPath = repositoryPath.resolveUserPathForContainer(ownerUid, calendarUid,
				RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION);
		try (InputStream in = Files.newInputStream(jsonPath, StandardOpenOption.READ)) {
			return new CalendarRestorableBackupItem<>(JsonReadWrite.readerUser().read(in), icsPath, histPath);
		}
	}

	private CalendarRestorableBackupItem<User> getRestorableUserCalendar(String domainUid, String ownerUid,
			String calendarUid) throws Exception {
		Optional<CalendarRestorableBackupItem<User>> restoredCal = readZipFilesForUserCal(domainUid, ownerUid,
				calendarUid);
		if (restoredCal.isPresent()) {
			return restoredCal.get();
		}

		return readJsonFilesForUserCal(ownerUid, calendarUid);
	}

	private CalendarRestorableBackupItem<ResourceDescriptor> getRestorableResourceCalendar(String domainUid,
			String ownerUid, String calendarUid) throws Exception {
		Optional<CalendarRestorableBackupItem<ResourceDescriptor>> restoredCal = readZipFilesForResourceCal(domainUid,
				ownerUid, calendarUid);
		if (restoredCal.isPresent()) {
			return restoredCal.get();
		}

		return readJsonFilesForResourceCal(ownerUid, calendarUid);
	}

	private Optional<CalendarRestorableBackupItem<ResourceDescriptor>> readZipFilesForResourceCal(String domainUid,
			String ownerUid, String calendarUid) throws Exception {
		return repositoryPath.getZipFilePathForOwner(domainUid, ownerUid).map(zipFilePath -> {
			return extractResourceCalendarDataFromZip(ownerUid, calendarUid, zipFilePath);
		});
	}

	private CalendarRestorableBackupItem<ResourceDescriptor> readJsonFilesForResourceCal(String ownerUid,
			String calendarUid) throws IOException {
		Path jsonPath = repositoryPath.resolveUserPathForContainer(ownerUid, calendarUid,
				RepositoryBackupPath.JSON_FILE_EXTENSION);
		Path icsPath = repositoryPath.resolveUserPathForContainer(ownerUid, calendarUid,
				RepositoryBackupPath.ICS_FILE_EXTENSION);
		Path histPath = repositoryPath.resolveUserPathForContainer(ownerUid, calendarUid,
				RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION);
		try (InputStream in = Files.newInputStream(jsonPath, StandardOpenOption.READ)) {
			return new CalendarRestorableBackupItem<>(JsonReadWrite.readerResource().read(in), icsPath, histPath);
		}

	}

	private record CalendarAllByteArrays(ByteBuf jsonData, ByteBuf icsData, ByteBuf historyData) {

	}

	private CalendarAllByteArrays readCalendarZip(String folderName, String calendarUid, Path zipFilePath) {

		Map<String, ByteBuf> filesFromZip = new HashMap<>();
		try {
			filesFromZip = JsonReadWrite.findFilesInZipFile(zipFilePath, folderName, calendarUid,
					Arrays.asList(RepositoryBackupPath.JSON_FILE_EXTENSION, RepositoryBackupPath.ICS_FILE_EXTENSION,
							RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION));

		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return new CalendarAllByteArrays(filesFromZip.get(RepositoryBackupPath.JSON_FILE_EXTENSION),
				filesFromZip.get(RepositoryBackupPath.ICS_FILE_EXTENSION),
				filesFromZip.get(RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION));

	}

	private CalendarRestorableBackupItem<User> extractUserCalendarDataFromZip(String folderName, String calendarUid,
			Path zipFilePath) {
		CalendarRestorableBackupItem<User> userCalendar = null;
		CalendarAllByteArrays calendarZip = readCalendarZip(folderName, calendarUid, zipFilePath);

		if (calendarZip.jsonData != null) {
			userCalendar = new CalendarRestorableBackupItem<>(JsonReadWrite.readerUser().read(calendarZip.jsonData));
			if (calendarZip.icsData != null) {
				userCalendar.readDataStreamContent(calendarZip.icsData);
			}
			if (calendarZip.historyData != null) {
				userCalendar.calendarHistory = new RestorableItemBackupItem<CalendarHistory>(
						JsonReadWrite.readerCalHistory().read(calendarZip.historyData));
			}
		}
		return userCalendar;
	}

	private CalendarRestorableBackupItem<ResourceDescriptor> extractResourceCalendarDataFromZip(String folderName,
			String calendarUid, Path zipFilePath) {
		CalendarRestorableBackupItem<ResourceDescriptor> userCalendar = null;
		CalendarAllByteArrays calendarZip = readCalendarZip(folderName, calendarUid, zipFilePath);

		if (calendarZip.jsonData != null) {
			userCalendar = new CalendarRestorableBackupItem<>(
					JsonReadWrite.readerResource().read(calendarZip.jsonData));
			if (calendarZip.icsData != null) {
				userCalendar.readDataStreamContent(calendarZip.icsData);
			}
			if (calendarZip.historyData != null) {
				userCalendar.calendarHistory = new RestorableItemBackupItem<CalendarHistory>(
						JsonReadWrite.readerCalHistory().read(calendarZip.historyData));
			}
		}
		return userCalendar;
	}

	public CalendarRestorableBackupItem<Domain> getRestorableDomainCalendar(String domainUid, String calendarUid)
			throws IOException {
		Path jsonPath = repositoryPath.resolveDomainPathForContainer(domainUid, calendarUid,
				RepositoryBackupPath.JSON_FILE_EXTENSION);
		Path icsPath = repositoryPath.resolveDomainPathForContainer(domainUid, calendarUid,
				RepositoryBackupPath.ICS_FILE_EXTENSION);
		Path histPath = repositoryPath.resolveUserPathForContainer(domainUid, calendarUid,
				RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION);
		try (InputStream in = Files.newInputStream(jsonPath, StandardOpenOption.READ)) {
			return new CalendarRestorableBackupItem<>(JsonReadWrite.readerDomain().read(in), icsPath, histPath);
		}
	}

}
