/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2023
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/
package net.bluemind.dataprotect.addressbook.impl.domain;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.AddressBookDescriptor;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.addressbook.impl.AbstractRestoreBook;
import net.bluemind.dataprotect.addressbook.impl.AddressBookBackupRepository;
import net.bluemind.dataprotect.addressbook.impl.CommonRestoreBooks;
import net.bluemind.dataprotect.addressbook.impl.pg.RestoreDomainBooksTaskPg;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.api.RestorableKind;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.domain.api.Domain;

public class RestoreDomainBooksTask extends AbstractRestoreBook implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(RestoreDomainBooksTask.class);

	protected final RestoreDomainBooksWorker addressBooksWorker;

	public RestoreDomainBooksTask(DataProtectGeneration backup, Restorable item) {
		super(backup, item);
		this.addressBooksWorker = new RestoreDomainBooksWorker(backup, restorableItem);
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startRestoreMsg(DPError.DPKind.DOMAIN_ADDRESSBOOK.name()));
		if (logger.isInfoEnabled()) {
			logger.info("Starting restore for uid {}", restorableItem.entryUid());
		}

		AddressBookBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new RestoreDomainBooksTaskPg(backup, restorableItem, addressBooksWorker).run();
		}
	}

	public void restore(AddressBookBackupRepository repository) {
		try {
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
			try (RestorableContainerItemBackupItem<Domain> restorable = repository
					.getRestorableDomainAddressBook(restorableItem.domain(), restorableItem.entryUid())) {
				ContainerDescriptor cd = restorable.backupDescriptor.getContainer();
				AddressBookDescriptor backupAddressBook = AddressBookDescriptor.create(cd.name, cd.owner, cd.domainUid,
						cd.settings);

				List<ItemValue<VCard>> vcards = importVcf(restorable, cd.owner, cd.domainUid,
						RestorableKind.ADDRESSBOOK);
				restorableItem.monitorLog("Restore domain addressbook {}@{} ({})", backupAddressBook.name,
						backupAddressBook.domainUid, restorableItem.entryUid());

				addressBooksWorker.createOrUpdateLiveEntry(ItemValue.create(cd.uid, backupAddressBook));

				new CommonRestoreBooks(restorableItem, live).restoreEntities(vcards, restorableItem.liveEntryUid());

			}
		} catch (Exception e) {
			logger.error("Error while restoring domain addressbook {}@{}: {}", restorableItem.entryUid(),
					restorableItem.domain(), e.getMessage());
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.DOMAIN_ADDRESSBOOK));
		} finally {
			restorableItem.endTask();
		}
	}
}
