/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.addressbook.impl.pg;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.IAddressBook;
import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.dataprotect.addressbook.impl.CommonRestoreBooks;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;

public class RestoreUserBooksTaskPg {
	private static final Logger logger = LoggerFactory.getLogger(RestoreUserBooksTaskPg.class);
	private IMonitoredRestoreRestorableItem restorableItem;
	private DataProtectGeneration backup;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@Deprecated
	public RestoreUserBooksTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem restorableItem) {
		this.backup = backup;
		this.restorableItem = restorableItem;
	}

	public void run() {
		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext back = bdp.createContextWithData(backup, restorableItem.item());
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();

			List<ContainerDescriptor> backABs = back.provider().instance(IContainers.class)
					.all(ContainerQuery.ownerAndType(restorableItem.entryUid(), IAddressBookUids.TYPE));

			restorableItem.monitorLog("Restoring '{}' addressbook(s) from uid={}", backABs.size(),
					restorableItem.entryUid());
			logger.info("Backup contains {} addressbook(s)", backABs.size());

			CommonRestoreBooks restoreBooks = new CommonRestoreBooks(restorableItem, back, live);
			for (ContainerDescriptor backAB : backABs) {
				restorableItem.monitorSubWork(1);
				dbrestore(restoreBooks, backAB);
			}
		} catch (Exception e) {
			logger.error("Error while restoring addressbooks", e);
			restorableItem.errors()
					.add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.ADDRESSBOOK));
		} finally {
			restorableItem.endTask();
		}
	}

	private void dbrestore(CommonRestoreBooks restoreBooks, ContainerDescriptor backAB) {
		IAddressBook backupABApi = restoreBooks.back.provider().instance(IAddressBook.class, backAB.uid);
		List<String> allUids = backupABApi.allUids();
		restorableItem.monitorLog("Restoring {} [uid={}]", backAB.name, backAB.uid);
		String bookUid = mapBookUid(backAB.uid);
		reset(restoreBooks, backAB, bookUid);
		restoreBooks.restoreEntities(allUids, backAB.uid, bookUid);
	}

	private String mapBookUid(String uid) {
		if (!restorableItem.entryUid().equals(restorableItem.liveEntryUid())
				&& uid.endsWith(String.format("_%s", restorableItem.entryUid()))) {
			return String.format("%s%s", uid.substring(0, uid.length() - restorableItem.entryUid().length()),
					restorableItem.liveEntryUid());
		}
		return uid;
	}

	private void reset(CommonRestoreBooks restoreBooks, ContainerDescriptor backAB, String bookUid) {
		List<ContainerDescriptor> liveABs = restoreBooks.live.provider().instance(IContainers.class)
				.all(ContainerQuery.ownerAndType(restorableItem.liveEntryUid(), IAddressBookUids.TYPE));

		if (liveABs.stream().noneMatch(c -> c.uid.equals(bookUid))) {
			backAB.owner = restorableItem.liveEntryUid();
			restoreBooks.live.provider().instance(IContainers.class).create(bookUid, backAB);
			restorableItem.monitorLog("addressbook recreated");
		}
	}
}
