/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.addressbook.impl.domain;

import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.AddressBookDescriptor;
import net.bluemind.addressbook.api.IAddressBooksMgmt;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.directory.RestoreDirWorker;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;

public class RestoreDomainBooksWorker extends RestoreDirWorker<AddressBookDescriptor> {
	private static final Logger logger = LoggerFactory.getLogger(RestoreDomainBooksWorker.class);

	public RestoreDomainBooksWorker(DataProtectGeneration backup, IMonitoredRestoreRestorableItem restorableItem) {
		super(backup, restorableItem);
	}

	@Override
	public ItemValue<AddressBookDescriptor> createOrUpdateLiveEntry(ItemValue<AddressBookDescriptor> backupResource)
			throws InterruptedException {

		BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
		IAddressBooksMgmt serviceLive = live.provider().instance(IAddressBooksMgmt.class);

		AddressBookDescriptor liveEntry = serviceLive.getComplete(restorableItem.liveEntryUid());
		if (liveEntry != null) {
			serviceLive.update(restorableItem.liveEntryUid(), backupResource.value);
			restorableItem.setLiveEntryUid(restorableItem.liveEntryUid());
			return ItemValue.create(restorableItem.liveEntryUid(), liveEntry);
		}

		DirEntryQuery query = DirEntryQuery.filterName(backupResource.value.name);
		query.kindsFilter = Arrays.asList(Kind.ADDRESSBOOK);
		query.systemFilter = false;

		IDirectory dir = live.provider().instance(IDirectory.class, backupResource.value.domainUid);
		ListResult<ItemValue<DirEntry>> booksList = dir.search(query);
		if (booksList.total > 1) {
			throw new ServerFault(
					String.format("More than one domain addressbook named '%s' found", backupResource.value.name));
		}
		booksList.values.forEach(de -> {
			if (de != null) {
				serviceLive.delete(de.uid);
			}
		});

		backupResource.value.owner = backupResource.value.domainUid;
		serviceLive.create(restorableItem.entryUid(), backupResource.value, false);
		liveEntry = serviceLive.getComplete(restorableItem.entryUid());
		restorableItem.setLiveEntryUid(restorableItem.entryUid());
		return ItemValue.create(restorableItem.liveEntryUid(), liveEntry);
	}

	@Override
	public void restoreEntryDependantContainers() {
	}
}
