/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.addressbook.impl.user;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.addressbook.impl.AbstractRestoreBook;
import net.bluemind.dataprotect.addressbook.impl.AddressBookBackupRepository;
import net.bluemind.dataprotect.addressbook.impl.pg.SendUserBookVCFTaskPg;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.email.SendVcf;
import net.bluemind.dataprotect.common.restore.MonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;
import net.bluemind.user.api.User;

public class SendUserBooksVCFTask extends AbstractRestoreBook implements IServerTask {
	private static final Logger logger = LoggerFactory.getLogger(SendUserBooksVCFTask.class);

	private final RestoreActionExecutor<EmailData> executor;
	private final ResourceBundle bundle;

	@SuppressWarnings("unchecked")
	public SendUserBooksVCFTask(DataProtectGeneration backup, Restorable item,
			RestoreActionExecutor<? extends IRestoreActionData> executor) {
		super(backup, item);
		this.executor = (RestoreActionExecutor<EmailData>) executor;
		this.bundle = ResourceBundle.getBundle("OSGI-INF/l10n/RestoreAddressBook", Locale.of(ServerSideServiceProvider
				.getProvider(SecurityContext.SYSTEM).getContext().getSecurityContext().getLang()));
	}

	@Override
	public void run(IServerTaskMonitor monitor) throws Exception {
		restorableItem = new MonitoredRestoreRestorableItem(restorableItem.item(), monitor);
		restorableItem.monitorBegin(1, restorableItem.startExportMsg(DPError.DPKind.ADDRESSBOOK.name()));
		if (logger.isInfoEnabled()) {
			logger.info("Starting export for uid {}", restorableItem.entryUid());
		}

		AddressBookBackupRepository repository = getRepository();
		if (repository != null) {
			restore(repository);
		} else {
			new SendUserBookVCFTaskPg(backup, bundle, restorableItem, executor).run();
		}

	}

	public void restore(AddressBookBackupRepository repository) throws Exception {
		SendVcf sendEmail = new SendVcf(restorableItem, executor);
		try {
			List<String> addressBookUids = repository.getRestorableUserAddressBookUids(restorableItem.domain(),
					restorableItem.entryUid());
			logger.info("AddressBook uids: {}", addressBookUids);
			Map<String, String> allVCards = HashMap.newHashMap(addressBookUids.size());
			for (String addressBookUid : addressBookUids) {
				try (RestorableContainerItemBackupItem<User> restorable = repository.getRestorableUserAddressBook(
						restorableItem.domain(), restorableItem.entryUid(), addressBookUid)) {
					ContainerDescriptor cd = restorable.backupDescriptor.getContainer();
					allVCards.put(cd.name, new String(restorable.dataStream().readAllBytes()));
				} catch (Exception e) {
					logger.error("Error while export user addressbook", e);
					restorableItem.errors()
							.add(DPError.export(e.getMessage(), addressBookUid, DPError.DPKind.ADDRESSBOOK));
				}
			}
			sendEmail.sendMessage(allVCards, bundle.getString("send.addressBook.restore.message"),
					bundle.getString("send.addressBook.restore.subject"));
		} catch (Exception e) {
			logger.error("Error while export user addressbook", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.ADDRESSBOOK));
		} finally {
			restorableItem.endTask();
		}
	}

}
