/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.addressbook.impl;

import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.common.backup.BackupContainerItemDescriptor;
import net.bluemind.dataprotect.common.backup.JsonReadWrite;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.domain.api.Domain;
import net.bluemind.user.api.User;

public class AddressBookBackupRepository {
	public static final Path DEFAULT_PATH = Paths.get("/var/backups/bluemind/addressbooks");
	private static Logger logger = LoggerFactory.getLogger(AddressBookBackupRepository.class);
	RepositoryBackupPath repositoryPath;

	public AddressBookBackupRepository(Path repository) {
		this.repositoryPath = new RepositoryBackupPath(repository);
	}

	record BackupUserAddressbookRecord(ContainerDescriptor cd, ItemValue<User> user, ByteBuf vcf) {
	}

	public List<String> writeUserAddressbooks(List<BackupUserAddressbookRecord> adBooksToBackup) throws IOException {
		List<String> ret = new ArrayList<>();
		for (BackupUserAddressbookRecord book : adBooksToBackup) {
			ret.add(writeUserAddressBookDescriptor(book));
			ret.add(writeContainerDescriptor(book.cd, book.vcf));
		}
		return ret;
	}

	private void writeDomainAddressBookDescriptor(ContainerDescriptor containerDescriptor, ItemValue<Domain> domain)
			throws IOException {
		Path jsonPathToWrite = repositoryPath.prepareJsonPathToWrite(containerDescriptor);
		JsonReadWrite.writeDomain(jsonPathToWrite,
				new BackupContainerItemDescriptor<Domain>(containerDescriptor, domain));
	}

	private String writeUserAddressBookDescriptor(BackupUserAddressbookRecord addBookRecord) throws IOException {
		Path jsonPathToWrite = repositoryPath.prepareJsonPathToWrite(addBookRecord.cd);
		JsonReadWrite.writeUser(jsonPathToWrite,
				new BackupContainerItemDescriptor<User>(addBookRecord.cd, addBookRecord.user));
		return RepositoryBackupPath.getFilePathWithDirectParent(jsonPathToWrite);
	}

	public void writeDomainAddressBook(ContainerDescriptor containerDescriptor, ItemValue<Domain> domain,
			ByteBuf exportString) throws IOException {
		writeDomainAddressBookDescriptor(containerDescriptor, domain);
		writeContainerDescriptor(containerDescriptor, exportString);
	}

	private String writeContainerDescriptor(ContainerDescriptor containerDescriptor, ByteBuf vcf) throws IOException {
		Path jsonPathToWrite = repositoryPath.getStoragePath(containerDescriptor,
				RepositoryBackupPath.VCF_FILE_EXTENSION);
		try (SeekableByteChannel out = Files.newByteChannel(jsonPathToWrite, StandardOpenOption.CREATE,
				StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING)) {
			out.write(vcf.nioBuffer());
		}
		return RepositoryBackupPath.getFilePathWithDirectParent(jsonPathToWrite);
	}

	/**
	 * Returns the list of addressbook uids available in the backup
	 *
	 * @param ownerUid
	 * @return List of uid of addressbook
	 * @throws IOException
	 */
	public List<String> getRestorableUserAddressBookUids(String domainUid, String ownerUid) throws IOException {
		return repositoryPath.getRestorableUserContainerUids(domainUid, ownerUid);
	}

	private RestorableContainerItemBackupItem<User> extractUserAddBookDataFromZip(String folderName, String adBookUid,
			Path zipFilePath) {
		RestorableContainerItemBackupItem<User> userAbBook = null;
		try {
			Map<String, ByteBuf> filesFromZip = JsonReadWrite.findFilesInZipFile(zipFilePath, folderName, adBookUid,
					Arrays.asList(RepositoryBackupPath.JSON_FILE_EXTENSION, RepositoryBackupPath.VCF_FILE_EXTENSION));

			ByteBuf jsonData = filesFromZip.get(RepositoryBackupPath.JSON_FILE_EXTENSION);
			if (jsonData != null) {
				userAbBook = new RestorableContainerItemBackupItem<>(JsonReadWrite.readerUser().read(jsonData));
				ByteBuf vcfData = filesFromZip.get(RepositoryBackupPath.VCF_FILE_EXTENSION);
				if (vcfData != null) {
					userAbBook.readDataStreamContent(vcfData);
				}
			}
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return userAbBook;
	}

	private Optional<RestorableContainerItemBackupItem<User>> readZipFilesForUserAddBook(String domainUid,
			String ownerUid, String addressbookUid) throws Exception {
		return repositoryPath.getZipFilePathForOwner(domainUid, ownerUid).map(zipFilePath -> {
			return extractUserAddBookDataFromZip(ownerUid, addressbookUid, zipFilePath);
		});
	}

	private RestorableContainerItemBackupItem<User> readJsonFilesForUserAddBook(String ownerUid, String addressbookUid)
			throws IOException {
		Path jsonPath = repositoryPath.resolveUserPathForContainer(ownerUid, addressbookUid,
				RepositoryBackupPath.JSON_FILE_EXTENSION);
		Path vcfPath = repositoryPath.resolveUserPathForContainer(ownerUid, addressbookUid,
				RepositoryBackupPath.VCF_FILE_EXTENSION);
		try (InputStream in = Files.newInputStream(jsonPath, StandardOpenOption.READ)) {
			return new RestorableContainerItemBackupItem<>(JsonReadWrite.readerUser().read(in.readAllBytes()), vcfPath);
		}
	}

	public RestorableContainerItemBackupItem<User> getRestorableUserAddressBook(String domainUid, String ownerUid,
			String addressbookUid) throws Exception {
		Optional<RestorableContainerItemBackupItem<User>> restoredAddBook = readZipFilesForUserAddBook(domainUid,
				ownerUid, addressbookUid);
		if (restoredAddBook.isPresent()) {
			return restoredAddBook.get();
		}

		return readJsonFilesForUserAddBook(ownerUid, addressbookUid);
	}

	public RestorableContainerItemBackupItem<Domain> getRestorableDomainAddressBook(String domainUid,
			String addressbookUid) throws IOException {
		Path jsonPath = repositoryPath.resolveDomainPathForContainer(domainUid, addressbookUid,
				RepositoryBackupPath.JSON_FILE_EXTENSION);
		Path vcfPath = repositoryPath.resolveDomainPathForContainer(domainUid, addressbookUid,
				RepositoryBackupPath.VCF_FILE_EXTENSION);
		try (InputStream in = Files.newInputStream(jsonPath, StandardOpenOption.READ)) {
			return new RestorableContainerItemBackupItem<>(JsonReadWrite.readerDomain().read(in.readAllBytes()),
					vcfPath);
		}
	}
}
