/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cql.testhelper;

import java.util.List;
import java.util.Objects;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.cql.CqlKeyspaceService;
import net.bluemind.cql.CqlSchemaService;
import net.bluemind.cql.CqlSessions;
import net.bluemind.cql.persistence.CqlKeyspaceStore;
import net.bluemind.cql.persistence.CqlSchemaStore;
import net.bluemind.dockerclient.DockerEnv;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public class CqlTestHelper {

	static {
		System.setProperty("cql.unsafe.for.tests", "true");
		System.setProperty("cql.flavor", "true");
	}

	private CqlTestHelper() {
	}

	public static void beforeClass() {
		String cassIp = DockerEnv.getIp("bluemind/scylladb");
		Objects.requireNonNull(cassIp, "Scylla docker IP is null, is container running ?");
		updateTopo(cassIp);
	}

	private static void updateTopo(String cassIp) {
		Server srv = new Server();
		srv.tags = List.of(TagDescriptor.cql_node.getTag());
		srv.ip = cassIp;
		ItemValue<Server> cassServerItem = ItemValue.create("cassandra", srv);
		Topology.update(List.of(cassServerItem));
		System.err.println("Topology updated with " + srv);
	}

	public static void beforeTest() {
		updateTopo(DockerEnv.getIp("bluemind/scylladb"));
		try (CqlSession systemSession = CqlSessions.system()) {
			CqlKeyspaceStore store = new CqlKeyspaceStore(systemSession);
			CqlKeyspaceService service = new CqlKeyspaceService(store);
			List<String> spaces = service.initialize();
			for (String ks : spaces) {
				CqlSchemaStore ss = new CqlSchemaStore(CqlSessions.forKeyspace(ks));
				CqlSchemaService schemas = new CqlSchemaService(ks, ss);
				schemas.initialize();
			}
		}
	}

	public static void afterTest() {
	}

}
