package net.bluemind.cql.node.service;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Splitter;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.cql.node.api.ICqlNode;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.network.topology.Topology;
import net.bluemind.node.api.ExitList;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NCUtils;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public class CqlNodeService implements ICqlNode {

	private static final Logger logger = LoggerFactory.getLogger(CqlNodeService.class);
	private final BmContext ctx;

	public CqlNodeService(BmContext ctx) {
		this.ctx = ctx;
		logger.debug("Using context {}", this.ctx);
	}

	public void addNode(ItemValue<Server> freshNode) {
		try {
			List<ItemValue<Server>> fullList = Topology.get().all(TagDescriptor.cql_node.getTag()).stream()
					.filter(s -> !s.uid.equals(freshNode.uid)).toList();
			List<ItemValue<Server>> withFresh = new ArrayList<>(fullList);
			withFresh.add(freshNode);
			configureSeeds(withFresh);
		} catch (Exception e) {
			logger.error("CQL {}", e.getMessage(), e);
		}
	}

	private void configureSeeds(List<ItemValue<Server>> fullList) {
		logger.info("Configuring seeds for {} server(s)", fullList.size());
		for (ItemValue<Server> current : fullList) {
			INodeClient nc = NodeActivator.get(current.value.address());
			if (nc.exists("/etc/scylla/scylla.yaml")) {
				String seeds = fullList.stream().map(other -> other.value.address() + ":7000")
						.collect(Collectors.joining(",", "          - seeds: \"", "\""));
				if (logger.isInfoEnabled()) {
					logger.info("Applying seeds to {}: {}", current.value.address(), seeds);
				}
				ExitList res = scyllaStrategy(seeds, nc);
				logger.info("Restarting scylla after update on {} => exit {}", current.value, res.getExitCode());
			} else {
				String seeds = fullList.stream().map(other -> other.value.address() + ":7000")
						.collect(Collectors.joining(",", "     - seeds: \"", "\""));
				if (logger.isInfoEnabled()) {
					logger.info("Applying seeds to {}: {}", current.value.address(), seeds);
				}
				ExitList res = cassandraStrategy(seeds, nc);
				logger.info("Restarting cassandra after update on {} => exit {}", current.value, res.getExitCode());
			}
			// for containers without systemctl support
			VertxPlatform.eventBus().publish("cql.node.cmd.restart", current.uid);
		}
	}

	private ExitList cassandraStrategy(String seeds, INodeClient nc) {
		byte[] tpl = nc.read("/usr/lib/bm-cassandra/conf/cassandra.yaml.tpl");
		byte[] updatedTemplate = Splitter.on('\n').splitToList(new String(tpl)).stream()
				.map(line -> line.contains("- seeds:") ? seeds : line).collect(Collectors.joining("\n")).getBytes();
		nc.writeFile("/usr/lib/bm-cassandra/conf/cassandra.yaml.tpl", new ByteArrayInputStream(updatedTemplate));
		return NCUtils.exec(nc, 30, TimeUnit.SECONDS, "systemctl", "restart", "bm-cassandra");
	}

	private ExitList scyllaStrategy(String seeds, INodeClient nc) {
		byte[] tpl = nc.read("/etc/scylla/scylla.yaml");
		byte[] updatedTemplate = Splitter.on('\n').splitToList(new String(tpl)).stream()
				.map(line -> line.contains("- seeds:") ? seeds : line).collect(Collectors.joining("\n")).getBytes();
		nc.writeFile("/etc/scylla/scylla.yaml", new ByteArrayInputStream(updatedTemplate));
		return NCUtils.exec(nc, 30, TimeUnit.SECONDS, "systemctl", "restart", "scylla-server");
	}

	public void removeNode(String uid) {
		List<ItemValue<Server>> fullList = Topology.get().all(TagDescriptor.cql_node.getTag()).stream()
				.filter(iv -> !iv.uid.equals(uid)).toList();
		configureSeeds(fullList);
	}

}
