/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.tx.wrapper.internal;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Policy;
import com.github.benmanes.caffeine.cache.stats.CacheStats;

import net.bluemind.core.tx.wrapper.TxEnabler;

public final class TxAwareCaffeineCache<K, V extends Object> implements Cache<K, V> {

	private final Cache<K, V> delegate;
	private static final Logger logger = LoggerFactory.getLogger(TxAwareCaffeineCache.class);

	private final ThreadLocal<Cache<K, V>> inTxView;

	public TxAwareCaffeineCache(Cache<K, V> parent) {
		this.delegate = parent;
		this.inTxView = ThreadLocal.withInitial(this::prepareCache);
	}

	private Cache<K, V> prepareCache() {
		Cache<K, V> overlay = Caffeine.newBuilder().build();
		TxEnabler.durableStorageAction(() -> onTxFinish(overlay));
		TxEnabler.recoveryAction(() -> onTxFinish(overlay));
		return overlay;
	}

	private void onTxFinish(Cache<K, V> overlay) {
		if (logger.isTraceEnabled()) {
			logger.trace("Forget {} key(s) set in transaction", overlay.estimatedSize());
		}
		var overlayKeys = overlay.asMap().keySet();
		if (!overlayKeys.isEmpty()) {
			delegate.invalidateAll(overlay.asMap().keySet());
		}
		overlay.invalidateAll();
		inTxView.remove();
	}

	public V getIfPresent(K key) {
		if (TxEnabler.isInTransaction()) {
			Cache<K, V> overlay = inTxView.get();
			return overlay.getIfPresent(key);
		}
		return delegate.getIfPresent(key);
	}

	public V get(K key, Function<? super K, ? extends V> mappingFunction) {
		if (TxEnabler.isInTransaction()) {
			V upstreamValue = delegate.getIfPresent(key);
			if (upstreamValue != null) {
				return upstreamValue;
			}
			Cache<K, V> overlay = inTxView.get();
			return overlay.get(key, mappingFunction::apply);
		}

		return delegate.get(key, mappingFunction);
	}

	public Map<K, V> getAllPresent(Iterable<? extends K> keys) {
		throw new UnsupportedOperationException();
	}

	@Override
	public Map<K, V> getAll(Iterable<? extends K> keys,
			Function<? super Set<? extends K>, ? extends Map<? extends K, ? extends V>> mappingFunction) {
		throw new UnsupportedOperationException();
	}

	public void put(K key, V value) {
		if (TxEnabler.isInTransaction()) {
			Cache<K, V> overlay = inTxView.get();
			overlay.put(key, value);
		} else {
			delegate.put(key, value);
		}
	}

	public void putAll(Map<? extends K, ? extends V> map) {
		if (TxEnabler.isInTransaction()) {
			Cache<K, V> overlay = inTxView.get();
			overlay.putAll(map);
		} else {
			delegate.putAll(map);
		}
	}

	public void invalidate(K key) {
		if (TxEnabler.isInTransaction()) {
			Cache<K, V> overlay = inTxView.get();
			overlay.invalidate(key);
			TxEnabler.recoveryAction(() -> delegate.invalidate(key));
			TxEnabler.durableStorageAction(() -> delegate.invalidate(key));
		} else {
			delegate.invalidate(key);
		}
	}

	public void invalidateAll(Iterable<? extends K> keys) {
		if (TxEnabler.isInTransaction()) {
			Cache<K, V> overlay = inTxView.get();
			overlay.invalidateAll(keys);
			TxEnabler.durableStorageAction(() -> delegate.invalidateAll(keys));
			TxEnabler.recoveryAction(() -> delegate.invalidateAll(keys));
		} else {
			delegate.invalidateAll(keys);
		}
	}

	public void invalidateAll() {
		if (TxEnabler.isInTransaction()) {
			Cache<K, V> overlay = inTxView.get();
			overlay.invalidateAll();
			TxEnabler.recoveryAction(delegate::invalidateAll);
			TxEnabler.durableStorageAction(delegate::invalidateAll);
		} else {
			delegate.invalidateAll();
		}
	}

	public long estimatedSize() {
		return delegate.estimatedSize();
	}

	public CacheStats stats() {
		return delegate.stats();
	}

	public ConcurrentMap<K, V> asMap() {
		throw new UnsupportedOperationException();
	}

	public void cleanUp() {
		delegate.cleanUp();
	}

	public Policy<K, V> policy() {
		return delegate.policy();
	}

}
