/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.container.service.internal;

import static net.bluemind.core.container.service.internal.ReadOnlyMode.checkWritable;
import static net.bluemind.core.jdbc.JdbcAbstractStore.doOrContinue;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.authentication.api.incore.IInCoreAuthentication;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.hooks.ContainersHooks;
import net.bluemind.core.container.hooks.IContainersHook;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ContainerModifiableDescriptor;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.IChangelogStore;
import net.bluemind.core.container.repository.IContainerPersonalSettingsStore;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerSettingsStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.container.repository.IContainerSyncStore;
import net.bluemind.core.container.sharding.Sharding;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.sanitizer.Sanitizer;
import net.bluemind.core.validator.Validator;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.i18n.labels.I18nLabels;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.network.topology.Topology;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.user.repository.IUserSubscriptionStore;

public class Containers implements IContainers {

	private static final Set<String> DATA_CONTAINER_TYPES = Sharding.containerTypes();
	private final SecurityContext securityContext;
	private final Sanitizer sanitizer;
	private final BmContext context;
	private Validator validator;

	private static final List<IContainersHook> cHooks = ContainersHooks.get();
	private static final Logger logger = LoggerFactory.getLogger(Containers.class);

	public Containers(BmContext context) {
		this.securityContext = context.getSecurityContext();
		this.context = context;
		sanitizer = new Sanitizer(context);
		validator = new Validator(context);
	}

	@Override
	public BaseContainerDescriptor create(String uid, ContainerDescriptor descriptor) throws ServerFault {
		checkWritable();

		sanitizer.create(descriptor);
		validator.create(descriptor);

		if (!securityContext.isDomainGlobal()) {
			if (descriptor.domainUid == null) {
				throw new ServerFault("Only admin0 can create container without domain", ErrorCode.FORBIDDEN);
			} else if (!securityContext.isDomainAdmin(descriptor.domainUid)
					&& !securityContext.getSubject().equals(descriptor.owner)) {
				throw new ServerFault("User cannot create container without being the owner", ErrorCode.FORBIDDEN);
			}

		}
		Container container = Container.create(uid, descriptor.type, descriptor.name, descriptor.owner,
				descriptor.domainUid, descriptor.defaultContainer);
		container.readOnly = descriptor.readOnly;

		DataLocation loc = DataLocation.directory();
		if (!"global.virt".equals(container.domainUid) && DATA_CONTAINER_TYPES.contains(descriptor.type)
				&& !pfDirEntry(descriptor.owner, descriptor.domainUid)) {
			DirEntry entry = context.su().provider().instance(IDirectory.class, container.domainUid)
					.findByEntryUid(descriptor.owner);

			if (entry != null) {
				// domain OR domain AB
				if (entry.kind == Kind.DOMAIN || (entry.kind == Kind.ADDRESSBOOK && descriptor.defaultContainer
						&& descriptor.uid.equals("addressbook_" + container.domainUid))) {
					loc = DataLocation.directory();
				} else if (entry.dataLocation != null) {
					loc = DataLocation.of(entry.dataLocation);
				} else {
					throw new ServerFault("Failed to create container " + container + " : no datalocation for owner "
							+ entry.entryUid);
				}
			} else {
				throw new ServerFault("Failed to create container " + container + " : owner not found");
			}

		}

		DataLocation unchanged = loc;
		IContainerStore cs = RepositoryProvider.instance(IContainerStore.class, context, loc);
		Container reloaded = ServerFault.onException(() -> cs.create(container), ErrorCode.SQL_ERROR);
		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
		ServerFault.onExceptionVoid(() -> routeStore.storeRoute(reloaded.uid, unchanged), ErrorCode.SQL_ERROR);

		for (IContainersHook ch : cHooks) {
			try {
				ch.onContainerCreated(context, descriptor);
			} catch (Exception e) {
				logger.warn("{} onContainerCreated: {}", ch, e.getMessage());
			}
		}
		return reloaded.asDescriptor(loc.serverUid());
	}

	private static final Map<String, String> domainToPF = new ConcurrentHashMap<>();

	/**
	 * This is similar to PublicFolders#mailboxGuid but we don't want to depend on
	 * that.
	 *
	 * @param owner
	 * @param domainUid
	 * @return
	 */
	private boolean pfDirEntry(String owner, String domainUid) {
		return owner != null && domainUid != null && owner.equals(
				domainToPF.computeIfAbsent(domainUid, dom -> UUID.nameUUIDFromBytes(dom.getBytes()).toString()));
	}

	@Override
	public void delete(String uid) throws ServerFault {
		checkWritable();
		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
		ContainerUid cUid = ContainerUid.of(uid);
		DataLocation dataLoc = routeStore.routeOf(cUid);
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, dataLoc);

		Container container = doOrContinue("onContainerDeleted", () -> containerStore.get(uid));
		if (container == null) {
			logger.warn("container {} not found, cannot delete it", uid);
			return;
		}

		RBACManager.forContext(context).forContainer(container).check(Verb.Manage.name());

		ContainerDescriptor prev = asDescriptor(container, null);

		// t_container_sub lives in directory db
		IContainerStore dirContStore = RepositoryProvider.instance(IContainerStore.class, context,
				DataLocation.directory());
		List<String> subs = ServerFault.onException(() -> dirContStore.listSubscriptions(container),
				ErrorCode.SQL_ERROR);

		// We call the hooks before otherwise we loose the infos to retry hooks
		// if something fails
		if (!subs.isEmpty()) {
			logger.info("Removing {} subscription(s) to {}", subs.size(), uid);
			for (IContainersHook ch : cHooks) {
				ch.onContainerSubscriptionsChanged(context, prev, Collections.emptyList(), subs);
			}
		}

		ServerFault.onExceptionVoid(() -> dirContStore.deleteAllSubscriptions(container), ErrorCode.SQL_ERROR);

		IContainerPersonalSettingsStore personalSettingsStore = RepositoryProvider
				.instance(IContainerPersonalSettingsStore.class, context, container);
		IContainerSettingsStore settingsStore = RepositoryProvider.instance(IContainerSettingsStore.class, context,
				container);
		IContainerSyncStore syncStore = RepositoryProvider.instance(IContainerSyncStore.class, context, container);
		AclService aclService = new AclService(context, context.getSecurityContext(), container);

		ServerFault.onExceptionVoid(() -> {
			RepositoryProvider.instance(IChangelogStore.class, context, container).deleteLog();
			personalSettingsStore.deleteAll();
			settingsStore.delete();
			syncStore.suspendSync();
			aclService.deleteAll();
			routeStore.deleteRoute(cUid);
			containerStore.delete(container);
		}, ErrorCode.SQL_ERROR);

		for (IContainersHook ch : cHooks) {
			doOrContinue("onContainerDeleted", () -> {
				ch.onContainerDeleted(context, prev);
				return null;
			});
		}

		eventProducer().changed(prev.type, uid);
	}

	@Override
	public void update(String uid, ContainerModifiableDescriptor descriptor) throws ServerFault {
		checkWritable();
		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
		DataLocation dataLoc = routeStore.routeOf(ContainerUid.of(uid));
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, dataLoc);

		Container container = ServerFault.onException(() -> {
			Container c = containerStore.get(uid);
			ContainerDescriptor prev = ContainerDescriptor.create(c.uid, c.name, c.owner, c.type, c.domainUid, false);

			sanitizer.update(prev, descriptor);
			validator.update(prev, descriptor);

			RBACManager.forContext(context).forContainer(c).check(Verb.Manage.name());

			containerStore.update(uid, descriptor.name, c.defaultContainer);

			return c;
		}, ErrorCode.SQL_ERROR);

		ContainerDescriptor prev = ContainerDescriptor.create(container.uid, container.name, container.owner,
				container.type, container.domainUid, container.defaultContainer);

		ContainerDescriptor cur = ContainerDescriptor.create(container.uid, descriptor.name, container.owner,
				container.type, container.domainUid, descriptor.defaultContainer);
		cur.deleted = descriptor.deleted;

		for (IContainersHook ch : cHooks) {
			doOrContinue("onContainerUpdated", () -> {
				ch.onContainerUpdated(context, prev, cur);
				return null;
			});
		}

		eventProducer().changed(prev.type, uid);
	}

	@Override
	public List<ContainerDescriptor> getContainers(List<String> containerIds) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();

		// FIXME perm check is missing
		List<ContainerDescriptor> ret = new ArrayList<>(containerIds.size());
		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
		containerIds.forEach(uid -> {
			DataLocation dataLoc = routeStore.routeOf(ContainerUid.of(uid));
			IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, dataLoc);

			Optional.ofNullable(ServerFault.onException(() -> containerStore.get(uid), ErrorCode.SQL_ERROR))
					.ifPresent(c -> ret.add(asDescriptor(c, securityContext)));
		});

		return ret;
	}

	@Override
	public List<BaseContainerDescriptor> getContainersLight(List<String> containerIds) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();

		// FIXME perm check is missing
		List<BaseContainerDescriptor> ret = new ArrayList<>(containerIds.size());
		Map<String, IDirectory> dirApiByDomain = new HashMap<>();
		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);

		containerIds.forEach(uid -> {
			DataLocation dataLoc = routeStore.routeOf(ContainerUid.of(uid));
			IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, dataLoc);
			Optional.ofNullable(ServerFault.onException(() -> containerStore.get(uid), ErrorCode.SQL_ERROR))
					.ifPresent(c -> {
						IDirectory dirApi = c.domainUid != null ? dirApiByDomain.computeIfAbsent(c.domainUid,
								dom -> context.su().provider().instance(IDirectory.class, dom)) : null;
						ret.add(asDescriptorLight(c, securityContext, dirApi));
					});
		});

		return ret;
	}

	@Override
	public List<ContainerDescriptor> all(ContainerQuery query) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();

		List<ContainerDescriptor> ret = new ArrayList<>();
		ret.addAll(allContainers(mboxLocations(), query, this::isSharded));
		ret.addAll(allContainers(List.of(DataLocation.directory()), query, c -> true));
		return dedup(ret);
	}

	private List<DataLocation> mboxLocations() {
		return Topology.getIfAvailable()
				.map(tp -> tp.all(TagDescriptor.bm_pgsql_data.getTag(), TagDescriptor.mail_imap.getTag())//
						.stream().distinct().map(iv -> DataLocation.of(iv.uid)).toList())
				.orElseGet(Collections::emptyList);
	}

	private List<ContainerDescriptor> allContainers(Collection<DataLocation> dataSources, ContainerQuery query,
			Predicate<ContainerDescriptor> filter) {
		return dataSources.stream().flatMap(ds -> {
			List<ContainerDescriptor> containers = new ArrayList<>();
			IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, ds);

			if (query.owner != null && query.type != null) {
				// FIXME not everybody should be able to call this
				containers.addAll(asDescriptors(
						ServerFault.onException(() -> containerStore.findByContainerQuery(query), ErrorCode.SQL_ERROR),
						securityContext));
			} else {
				containers.addAll(asDescriptors(
						ServerFault.onException(() -> containerStore.findAccessiblesByType(query), ErrorCode.SQL_ERROR),
						securityContext));
			}
			return containers.stream().filter(filter);
		}).collect(Collectors.toList());
	}

	private boolean isSharded(BaseContainerDescriptor container) {
		return isShardedType(container.type);
	}

	private boolean isShardedContainer(Container container) {
		return isShardedType(container.type);
	}

	private boolean isShardedType(String type) {
		return Sharding.containerTypes().contains(type);
	}

	private static <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
		Set<Object> seen = ConcurrentHashMap.newKeySet();
		return t -> seen.add(keyExtractor.apply(t));
	}

	private static <W extends BaseContainerDescriptor> List<W> dedup(List<W> orig) {
		return orig.stream().filter(distinctByKey(c -> c.uid)).collect(Collectors.toList());
	}

	@Override
	public List<BaseContainerDescriptor> allLight(ContainerQuery query) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();

		List<Container> ret = new ArrayList<>();
		ret.addAll(queryContainer(query, mboxLocations(), this::isShardedContainer, securityContext));
		ret.addAll(queryContainer(query, List.of(DataLocation.directory()), c -> true, securityContext));

		return dedup(asDescriptorsLight(ret, securityContext));
	}

	@Override
	public List<ContainerDescriptor> allForUser(String domainUid, String userUid, ContainerQuery query)
			throws ServerFault {

		if (!context.getSecurityContext().isDomainAdmin(domainUid)) {
			throw new ServerFault("You have to be domain admin to call this method", ErrorCode.FORBIDDEN);
		}

		SecurityContext sc = context.provider().instance(IInCoreAuthentication.class).buildContext(domainUid, userUid);

		List<Container> ret = new ArrayList<>();
		ret.addAll(queryContainer(query, mboxLocations(), this::isShardedContainer, sc));
		ret.addAll(queryContainer(query, List.of(DataLocation.directory()), c -> true, sc));

		return dedup(asDescriptors(ret, sc));
	}

	private List<Container> queryContainer(ContainerQuery query, Collection<DataLocation> dataSources,
			Predicate<Container> filter, SecurityContext ctx) {
		List<Container> containers = new ArrayList<>();
		dataSources.forEach(ds -> {
			try {
				BmContext freshCtx = context.su(ctx);
				IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, freshCtx, ds);
				if (query.owner != null) {
					// FIXME not everybody should be able to call this
					containers.addAll(ServerFault.onException(() -> containerStore.findByContainerQuery(query),
							ErrorCode.SQL_ERROR));
				} else {
					containers.addAll(ServerFault.onException(() -> containerStore.findAccessiblesByType(query),
							ErrorCode.SQL_ERROR));
				}
			} catch (Exception e) {
				logger.warn("Fail to fetch containers on datasource {}", ds);
			}

		});
		return containers.stream().filter(filter).collect(Collectors.toList());
	}

	@Override
	public ContainerDescriptor getForUser(String domainUid, String userUid, String uid) throws ServerFault {

		if (!context.getSecurityContext().isDomainAdmin(domainUid)) {
			throw new ServerFault("You have to be domain admin to call this method", ErrorCode.FORBIDDEN);
		}

		SecurityContext userContext = context.provider().instance(IInCoreAuthentication.class).buildContext(domainUid,
				userUid);

		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
		DataLocation dataLoc = routeStore.routeOf(ContainerUid.of(uid));
		IContainerStore suContainerStore = RepositoryProvider.instance(IContainerStore.class, context, dataLoc);

		Container c = ServerFault.onException(() -> suContainerStore.get(uid), ErrorCode.SQL_ERROR);
		if (c == null) {
			throw new ServerFault("Container '" + uid + "' not found", ErrorCode.NOT_FOUND);
		}
		return asDescriptor(c, userContext);

	}

	public List<ContainerDescriptor> asDescriptors(List<Container> containers, SecurityContext sc) throws ServerFault {
		List<ContainerDescriptor> ret = new ArrayList<>(containers.size());
		for (Container c : containers) {
			ret.add(asDescriptor(c, sc));
		}
		return ret;
	}

	private List<BaseContainerDescriptor> asDescriptorsLight(List<Container> containers, SecurityContext sc)
			throws ServerFault {
		List<BaseContainerDescriptor> ret = new ArrayList<>(containers.size());
		Map<String, IDirectory> dirApiByDomain = new HashMap<>();
		for (Container c : containers) {
			IDirectory dirApi = c.domainUid != null
					? dirApiByDomain.computeIfAbsent(c.domainUid,
							dom -> context.su().provider().instance(IDirectory.class, dom))
					: null;
			ret.add(asDescriptorLight(c, sc, dirApi));
		}
		return ret;
	}

	BaseContainerDescriptor asDescriptorLight(Container c, SecurityContext sc, IDirectory dirApi) throws ServerFault {

		if (logger.isDebugEnabled()) {
			logger.debug("c: {}, context: {}", c, context);
		}
		String label = I18nLabels.getInstance().translate(sc.getLang(), c.name);
		BaseContainerDescriptor descriptor = BaseContainerDescriptor.create(c.uid, label, c.owner, c.type, c.domainUid,
				c.defaultContainer);

		if (descriptor.owner != null && dirApi != null) {
			try {
				DirEntry entry = dirApi.findByEntryUid(descriptor.owner);
				if (entry != null) {
					descriptor.ownerDisplayname = entry.displayName;
					descriptor.ownerDirEntryPath = entry.path;
				}
			} catch (Exception e) {
				logger.warn("error loading entry {}@{}", descriptor.owner, descriptor.domainUid, e);
			}
		}

		descriptor.readOnly = c.readOnly;
		descriptor.internalId = c.id;
		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
		descriptor.datalocation = routeStore.routeOf(ContainerUid.of(c.uid)).serverUid();

		IContainerSettingsStore settingsStore = RepositoryProvider.instance(IContainerSettingsStore.class, context, c);
		ServerFault.onException(() -> {
			descriptor.settings = settingsStore.getSettings();
			if (descriptor.settings == null) {
				descriptor.settings = Collections.emptyMap();
			}
			return null;
		}, ErrorCode.SQL_ERROR);
		return descriptor;
	}

	private ContainerDescriptor asDescriptor(Container c, SecurityContext sc) throws ServerFault {
		String label = c.name;
		if (sc != null) {
			label = I18nLabels.getInstance().translate(sc.getLang(), c.name);
		}
		ContainerDescriptor descriptor = ContainerDescriptor.create(c.uid, label, c.owner, c.type, c.domainUid,
				c.defaultContainer);
		descriptor.internalId = c.id;
		IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
		descriptor.datalocation = routeStore.routeOf(ContainerUid.of(c.uid)).serverUid();
		if (sc != null) {
			RBACManager aclForContainer = RBACManager.forSecurityContext(sc).forContainer(c);
			descriptor.verbs = aclForContainer.resolve().stream().filter(ContainerPermission.class::isInstance)
					.map(p -> Verb.valueOf(p.asRole())).collect(Collectors.toSet());
			descriptor.writable = descriptor.verbs.stream().anyMatch(verb -> verb.can(Verb.Write));

			IContainerStore cs = RepositoryProvider.instance(IContainerStore.class, context, DataLocation.directory());
			Container dom = ServerFault.onException(() -> cs.get(sc.getContainerUid()), ErrorCode.SQL_ERROR);
			IUserSubscriptionStore userSubscriptionStore = RepositoryProvider.instance(IUserSubscriptionStore.class,
					context, dom);

			descriptor.offlineSync = ServerFault
					.onException(() -> userSubscriptionStore.isSyncAllowed(sc.getSubject(), c), ErrorCode.SQL_ERROR);

		}

		if (descriptor.owner != null && descriptor.domainUid != null) {
			try {
				DirEntry entry = context.su().provider().instance(IDirectory.class, c.domainUid)
						.findByEntryUid(descriptor.owner);
				if (entry != null) {
					descriptor.ownerDisplayname = entry.displayName;
					descriptor.ownerDirEntryPath = entry.path;
				}
			} catch (Exception e) {
				logger.warn("error loading entry {}@{} for enhancing descriptor", descriptor.owner,
						descriptor.domainUid);
			}
		}

		descriptor.readOnly = c.readOnly;

		IContainerSettingsStore settingsStore = RepositoryProvider.instance(IContainerSettingsStore.class, context, c);
		IContainerPersonalSettingsStore personalSettingsStore = RepositoryProvider
				.instance(IContainerPersonalSettingsStore.class, context, c);
		ServerFault.onException(() -> {
			descriptor.settings = settingsStore.getSettings();
			if (descriptor.settings == null) {
				descriptor.settings = new HashMap<>();
			}
			Map<String, String> psettings = personalSettingsStore.get();
			if (psettings != null) {
				descriptor.settings.putAll(psettings);
			}

			// Filter password settings
			List<String> obfuscateSettings = List.of("loginPw");
			descriptor.settings = descriptor.settings.entrySet().stream()
					.filter(e -> !obfuscateSettings.contains(e.getKey()))
					.collect(Collectors.toMap(Entry::getKey, Entry::getValue));
			return null;
		}, ErrorCode.SQL_ERROR);
		return descriptor;
	}

	public ContainerDescriptor asDescriptorForUser(Container c, SecurityContext sc, String userUid) throws ServerFault {
		IContainerStore cs = RepositoryProvider.instance(IContainerStore.class, context, DataLocation.directory());
		Container dom = ServerFault.onException(() -> cs.get(sc.getContainerUid()), ErrorCode.SQL_ERROR);

		ContainerDescriptor ret = asDescriptor(c, sc);

		IUserSubscriptionStore userSubscriptionStore = RepositoryProvider.instance(IUserSubscriptionStore.class,
				context, dom);
		ret.offlineSync = ServerFault.onException(() -> userSubscriptionStore.isSyncAllowed(userUid, c),
				ErrorCode.SQL_ERROR);
		return ret;
	}

	@Override
	public ContainerDescriptor get(String uid) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();
		// FIXME we should check that, at least container.domainUid ==
		// user.domainUid
		// or domain admin
		ContainerUid cuid = ContainerUid.of(uid);
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, cuid);
		Container c = ServerFault.onException(() -> containerStore.get(uid), ErrorCode.SQL_ERROR);
		if (c == null && onDirectoryDatalocation(cuid)) {
			c = checkMboxDs(uid);
		}
		if (c == null) {
			throw ServerFault.notFound("Container '" + uid + "' not found");
		}
		return asDescriptor(c, securityContext);
	}

	private boolean onDirectoryDatalocation(ContainerUid cuid) {
		IContainerRouteStore route = RepositoryProvider.instance(IContainerRouteStore.class, context);
		DataLocation loc = route.routeOf(cuid);
		return loc == null || loc.equals(DataLocation.directory());
	}

	@Override
	public BaseContainerDescriptor getLight(String uid) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context,
				ContainerUid.of(uid));
		Container c = ServerFault.onException(() -> containerStore.get(uid), ErrorCode.SQL_ERROR);
		if (c == null && onDirectoryDatalocation(ContainerUid.of(uid))) {
			c = checkMboxDs(uid);
		}
		if (c == null) {
			throw ServerFault.notFound("Container '" + uid + "' not found");
		}
		return asDescriptorLight(c, securityContext, null);
	}

	@Override
	public BaseContainerDescriptor getLightIfPresent(String uid) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context,
				ContainerUid.of(uid));
		Container c = ServerFault.onException(() -> containerStore.get(uid), ErrorCode.SQL_ERROR);
		if (c == null && onDirectoryDatalocation(ContainerUid.of(uid))) {
			c = checkMboxDs(uid);
		}

		if (c == null) {
			// repair failed or container really does not exists.
			return null;
		}
		return asDescriptorLight(c, securityContext, null);
	}

	private Container checkMboxDs(String uid) {
		// Auto repair on get if containerLocation is not set
		if (mboxLocations().isEmpty()) {
			return null;
		}
		for (DataLocation loc : mboxLocations()) {
			IContainerStore onLoc = RepositoryProvider.instance(IContainerStore.class, context, loc);
			Container reloaded = ServerFault.onException(() -> onLoc.get(uid), ErrorCode.SQL_ERROR);
			if (reloaded != null) {
				IContainerRouteStore route = RepositoryProvider.instance(IContainerRouteStore.class, context);
				ServerFault.onExceptionVoid(() -> route.storeRoute(uid, loc), ErrorCode.SQL_ERROR);
				return reloaded;
			}
		}
		return null;
	}

	@Override
	public void setAccessControlList(String uid, List<AccessControlEntry> entries) throws ServerFault {
		IContainerManagement mgmt = context.provider().instance(IContainerManagement.class, uid);
		mgmt.setAccessControlList(entries);
	}

	@Override
	public ContainerDescriptor getIfPresent(String uid) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context,
				ContainerUid.of(uid));

		Container c = ServerFault.onException(() -> containerStore.get(uid), ErrorCode.SQL_ERROR);
		if (c == null && onDirectoryDatalocation(ContainerUid.of(uid))) {
			c = checkMboxDs(uid);
		}
		if (c == null) {
			return null;
		}
		return asDescriptor(c, securityContext);

	}

	private ContainersEventProducer eventProducer() {
		return new ContainersEventProducer(context.getSecurityContext(), VertxPlatform.eventBus());
	}

	@Override
	public Map<String, List<AccessControlEntry>> getAccessControlLists(List<String> containerIds) throws ServerFault {
		RBACManager.forContext(context).checkNotAnoynmous();
		SecurityContext sc = context.getSecurityContext();
		Set<String> matchingOwners = Stream.of(List.of(sc.getOwnerPrincipal()), sc.getMemberOf()).flatMap(List::stream)
				.collect(Collectors.toSet());
		return containerIds.stream().collect(
				Collectors.toMap(uid -> uid, uid -> context.su().provider().instance(IContainerManagement.class, uid)
						.getAccessControlList().stream().filter(acl -> matchingOwners.contains(acl.subject)).toList()));
	}

	@Override
	public boolean exists(String containerUid) throws ServerFault {
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context,
				ContainerUid.of(containerUid));
		return ServerFault.onException(() -> containerStore.exists(containerUid), ErrorCode.SQL_ERROR);
	}

}
