/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.service.acl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.IInternalContainerManagement;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.AclDiff;
import net.bluemind.core.container.model.acl.AclDiff.AclStatus;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.service.internal.Containers;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.rules.DelegationRule;
import net.bluemind.notes.api.INoteUids;
import net.bluemind.todolist.api.ITodoUids;

public class SharedMailboxAclManager {
	private final Set<AccessControlEntry> previousAcl;
	private final Set<AccessControlEntry> currentAcl;
	private final Kind dirEntryKind;

	private final List<DelegatorHerited> readExtendedToRemove = new ArrayList<>();
	private final List<DelegatorHerited> readExtendedToAdd = new ArrayList<>();

	private static final List<String> containerTypes = Arrays.asList(ICalendarUids.TYPE, IAddressBookUids.TYPE,
			ITodoUids.TYPE, INoteUids.TYPE);

	public SharedMailboxAclManager(SharedMailboxUpdateAcl previous, SharedMailboxUpdateAcl current) {
		this.previousAcl = previous.acl();
		this.currentAcl = current.acl();
		this.dirEntryKind = current.dirEntryKind();
	}

	public SharedMailboxAclManager(Set<AccessControlEntry> previousAcl, Set<AccessControlEntry> currentAcl,
			Kind dirEntryKind) {
		this.previousAcl = previousAcl;
		this.currentAcl = currentAcl;
		this.dirEntryKind = dirEntryKind;
	}

	protected void updateDelegationAccess(BmContext context, Container container) {
		if (container.type.equals(IMailboxAclUids.TYPE) && dirEntryKind == Kind.SHARED_MAILBOX) {
			applyDelegationChanges();
			propagateSharedMailboxAcl(context, container);
		}
	}

	private void propagateSharedMailboxAcl(BmContext context, Container container) {
		propagateAclToContainers(context, container);
		delegateCalendar(context, container);
	}

	private void propagateAclToContainers(BmContext context, Container container) {
		IContainers containerService = new Containers(context);
		containerTypes.stream() //
				.map(containerType -> ContainerQuery.ownerAndType(container.owner, containerType))
				.flatMap(query -> containerService.all(query).stream())
				.forEach(c -> updateAcl(context, container.domainUid, c.uid));
	}

	private void delegateCalendar(BmContext context, Container container) {
		applyReadExtendedAcl(context, container);
		createDelegateMailboxRule(context, container);
	}

	private void applyReadExtendedAcl(BmContext context, Container container) {
		if (readExtendedToAdd.isEmpty() && readExtendedToRemove.isEmpty()) {
			return;
		}

		IContainers containerService = new Containers(context);
		List<ContainerDescriptor> calendars = containerService
				.all(ContainerQuery.ownerAndType(container.owner, ICalendarUids.TYPE));
		for (ContainerDescriptor containerDescriptor : calendars) {
			IInternalContainerManagement containerMgmtService = context.provider()
					.instance(IInternalContainerManagement.class, containerDescriptor.uid);
			List<AccessControlEntry> accessControlList = containerMgmtService.getAccessControlList();
			if (!readExtendedToAdd.isEmpty()) {
				accessControlList.addAll(readExtendedToAdd.stream()
						.map(d -> AccessControlEntry.create(d.subject(), Verb.ReadExtended)).toList());
			}
			if (!readExtendedToRemove.isEmpty()) {
				accessControlList.removeAll(readExtendedToRemove.stream()
						.map(d -> AccessControlEntry.create(d.subject(), Verb.ReadExtended)).toList());
			}
			containerMgmtService.setAccessControlList(accessControlList, false);
		}
	}

	private record DelegatorHerited(String subject, boolean canWrite) {

		@Override
		public final String toString() {
			return subject + " => " + canWrite;
		}

		@Override
		public int hashCode() {
			return Objects.hash(canWrite, subject);
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			DelegatorHerited other = (DelegatorHerited) obj;
			return canWrite == other.canWrite && Objects.equals(subject, other.subject);
		}

	}

	private void createDelegateMailboxRule(BmContext context, Container container) {
		IMailboxes mailboxApiForDelegation = context.su().provider().instance(IMailboxes.class, container.domainUid);

		List<String> imipToAdd = readExtendedToAdd.stream().filter(d -> d.canWrite()).map(s -> s.subject).toList();
		List<String> imipToRemove = readExtendedToRemove.stream().filter(d -> d.canWrite()).map(s -> s.subject)
				.toList();

		DelegationRule mailboxDelegationRule = mailboxApiForDelegation.getMailboxDelegationRule(container.owner);
		if (mailboxDelegationRule != null) {
			mailboxDelegationRule.delegateUids.addAll(imipToAdd);
			mailboxDelegationRule.delegateUids.removeAll(imipToRemove);
		} else if (!readExtendedToAdd.isEmpty()) {
			mailboxDelegationRule = new DelegationRule(ICalendarUids.defaultUserCalendar(container.owner), imipToAdd,
					container.owner, false, false);
		} else {
			return;
		}

		if (!imipToAdd.isEmpty() || !imipToRemove.isEmpty()) {
			mailboxApiForDelegation.setMailboxDelegationRule(container.owner, mailboxDelegationRule);
		}
	}

	private void updateAcl(BmContext context, String domainUid, String containerUid) {
		IInternalContainerManagement containerMgmtService = context.provider()
				.instance(IInternalContainerManagement.class, containerUid);
		List<AccessControlEntry> acls = sanitizeContainerAcl(containerMgmtService.getAccessControlList(), domainUid);
		containerMgmtService.setAccessControlList(acls, false);
	}

	private List<String> aclSubjectsToRemove() {
		return Stream //
				.concat(previousAcl.stream().map(ace -> ace.subject), currentAcl.stream().map(ace -> ace.subject)) //
				.collect(Collectors.toSet()) //
				.stream() //
				.flatMap(subject -> AclDiff.prepareAclDiff(subject, previousAcl, currentAcl).stream()) //
				.filter(diff -> diff.status() == AclStatus.REMOVED) //
				.map(AclDiff::subject).toList();
	}

	private List<AccessControlEntry> sanitizeContainerAcl(List<AccessControlEntry> acls, String domainUid) {
		List<String> aclToRemove = aclSubjectsToRemove();
		acls.removeIf(ace -> !ace.subject.equals(domainUid) && aclToRemove.contains(ace.subject));
		acls.addAll(currentAcl);
		return AccessControlEntry.compact(acls);
	}

	private static Map<String, List<Verb>> aclAsMap(Set<AccessControlEntry> acls) {
		return acls.stream().collect(Collectors.groupingBy(AccessControlEntry::getSubject,
				Collectors.mapping(AccessControlEntry::getVerb, Collectors.toList())));
	}

	private static List<DelegatorHerited> containsDelegation(Map<String, List<Verb>> acls) {
		return acls.entrySet().stream() //
				.filter(entry -> entry.getValue().stream() //
						.anyMatch(v -> v.can(Verb.SendAs) || v.can(Verb.SendOnBehalf)))
				.map(entry -> new DelegatorHerited(entry.getKey(), entry.getValue().stream() //
						.anyMatch(v -> v.can(Verb.Write))))
				.collect(Collectors.toList());
	}

	private void applyDelegationChanges() {
		List<DelegatorHerited> currentWithDelegation = containsDelegation(aclAsMap(currentAcl));
		List<DelegatorHerited> previousWithDelegation = containsDelegation(aclAsMap(previousAcl));

		Set<DelegatorHerited> added = new HashSet<>(currentWithDelegation);
		added.removeAll(previousWithDelegation);
		readExtendedToAdd.addAll(added);

		Set<DelegatorHerited> removed = new HashSet<>(previousWithDelegation);
		removed.removeAll(currentWithDelegation);
		readExtendedToRemove.addAll(removed);
	}

}
