package net.bluemind.core.container.service;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.ISharedContainers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.SharedContainer;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.ISharedContainerStore;
import net.bluemind.core.container.repository.ISharedContainerStore.SharedContainerStoreResult;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.container.sharding.Sharding;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.repository.provider.RepositoryProvider;

public class SharedContainersService implements ISharedContainers {

	private static final Logger logger = LoggerFactory.getLogger(SharedContainersService.class);

	private BmContext context;
	private IContainers contApi;
	private String ownerUid;
	private IDirectory dirApi;

	public SharedContainersService(BmContext context, IDirectory dirApi, String ownerUid) {
		this.context = context;
		this.ownerUid = ownerUid;
		this.contApi = context.su().provider().instance(IContainers.class);
		this.dirApi = dirApi;
	}

	@Override
	public List<SharedContainer> getSharedContainers(String type) throws ServerFault {

		DirEntry dirEntry = dirApi.findByEntryUid(ownerUid);
		ContainerQuery query = ContainerQuery.ownerAndType(ownerUid, type);
		List<BaseContainerDescriptor> containers = contApi.allLight(query);

		boolean allowed = context.getSecurityContext().isDomainGlobal()
				|| containers.stream().anyMatch(cont -> RBACManager.forContext(context).forContainer(cont.uid)
						.can(Verb.Visible.name(), Verb.Manage.name()));
		if (!allowed) {
			throw new ServerFault(String.format("%s@%s Doesnt have role %s ", //
					context.getSecurityContext().getSubject(), //
					context.getSecurityContext().getContainerUid(), //
					Verb.Visible.name()), ErrorCode.PERMISSION_DENIED);
		}
		List<Long> ids = containers.stream().map(c -> c.internalId).toList();

		DataLocation loc = Sharding.containerTypes().contains(type) ? DataLocation.of(dirEntry.dataLocation)
				: DataLocation.directory();

		ISharedContainerStore sharedContainerService = RepositoryProvider.instance(ISharedContainerStore.class, context,
				loc);
		try {
			List<SharedContainerStoreResult> results = sharedContainerService.mget(ids).stream()
					.filter(result -> !result.acl().subject.equals(ownerUid)).toList();
			Map<BaseContainerDescriptor, Set<AccessControlEntry>> map = new HashMap<>();
			results.forEach(result -> {
				var set = map.computeIfAbsent(result.desc(), k -> new HashSet<>());
				set.add(result.acl());

			});
			return map.entrySet().stream().map(e -> new SharedContainer(e.getKey(), e.getValue())).toList();

		} catch (SQLException e) {
			logger.error(e.getMessage());
			return new ArrayList<>();
		}
	}

}
