/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.service.internal;

import java.sql.SQLException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.api.IOwnerSubscriptionUids;
import net.bluemind.core.container.api.internal.IInternalOwnerSubscriptions;
import net.bluemind.core.container.api.internal.IInternalOwnerSubscriptionsMgmt;
import net.bluemind.core.container.hooks.ContainersHooks;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.repository.provider.RepositoryProvider;

public class InternalOwnerSubscriptionsMgmtService implements IInternalOwnerSubscriptionsMgmt {

	private static final Logger logger = LoggerFactory.getLogger(InternalOwnerSubscriptionsMgmtService.class);
	private final BmContext context;
	private final String ownerUid;
	private final String domainUid;

	public InternalOwnerSubscriptionsMgmtService(BmContext context, String ownerUid, String domainUid) {
		this.context = context;
		this.ownerUid = ownerUid;
		this.domainUid = domainUid;
	}

	@Override
	public void init() {
		IDirectory dirApi = context.provider().instance(IDirectory.class, domainUid);
		DirEntry entry = dirApi.findByEntryUid(ownerUid);
		if (entry == null) {
			logger.warn("Entry not found with uid {}", ownerUid);
			return;
		}
		if (entry.kind != Kind.USER || entry.system) {
			return;
		}
		DataLocation loc = null;
		logger.info("***** Owner subscriptions init for user {}: {} @: {}", entry.displayName, ownerUid, domainUid);
		if (entry.dataLocation == null) {
			loc = DataLocation.directory();
		} else {
			loc = DataLocation.of(entry.dataLocation);
		}

		String subsUid = IOwnerSubscriptionUids.getIdentifier(ownerUid, domainUid);

		IContainerStore shardContStore = RepositoryProvider.instance(IContainerStore.class, context, loc);
		Container subsCont = Container.create(subsUid, IOwnerSubscriptionUids.TYPE,
				ownerUid + "@" + domainUid + " subscriptions", ownerUid);
		subsCont.domainUid = domainUid;
		subsCont.defaultContainer = true;
		try {
			Container existing = shardContStore.get(subsUid);
			if (existing == null) {
				Container created = shardContStore.create(subsCont);
				ContainerDescriptor cd = ContainerDescriptor.create(created.uid, created.name, created.owner,
						created.type, created.domainUid, true);
				cd.datalocation = entry.dataLocation;
				ContainersHooks.get().forEach(ch -> ch.onContainerCreated(context.su(), cd));
			} else {
				logger.warn("container {} already exist", subsUid);
			}
			IContainerRouteStore routeStore = RepositoryProvider.instance(IContainerRouteStore.class, context);
			routeStore.storeRoute(ContainerUid.of(subsCont.uid), loc);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}
	}

	@Override
	public void delete() {
		String subsUid = IOwnerSubscriptionUids.getIdentifier(ownerUid, domainUid);
		IContainers contApi = context.provider().instance(IContainers.class);
		try {
			BaseContainerDescriptor cd = contApi.getLight(subsUid);
			IInternalOwnerSubscriptions subsApi = context.provider().instance(IInternalOwnerSubscriptions.class,
					cd.domainUid, cd.owner);
			subsApi.reset();
			contApi.delete(subsUid);
			logger.info("***** Owner subscriptions deleted for: {} @: {}", ownerUid, domainUid);
		} catch (ServerFault sf) {
			if (sf.getCode() == ErrorCode.NOT_FOUND) {
				// this is fine, we might not be dealing with a user
			} else {
				throw sf;
			}
		}
	}

}
