/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.service;

import java.sql.SQLException;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IOwnerSubscriptionUids;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.container.service.internal.InternalOwnerSubscriptionsService;
import net.bluemind.core.container.service.internal.OwnerSubscriptionsEventProducer;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.repository.provider.RepositoryProvider;

public abstract class CommonOwnerSubscriptionsServiceFactory<T> {

	public abstract Class<T> factoryClass();

	public T instance(BmContext context, String... params) throws ServerFault {
		if (params == null || params.length < 2) {
			throw new ServerFault("2 parameters expected, domainUid & ownerUid (got " + params + ")");
		}

		String domainUid = params[0];
		String ownerUid = params[1];
		String containerUid = IOwnerSubscriptionUids.getIdentifier(ownerUid, domainUid);

		DataLocation loc = RepositoryProvider.instance(IContainerRouteStore.class, context)
				.routeOf(ContainerUid.of(containerUid));
		// Don't enforce a sharded DB here. Normally, the container is sharded, but
		// due to unknown reasons, the container
		// sometimes is present in the directory database
		if (loc.equals(DataLocation.directory())) {
			DirEntry entry = context.su().provider().instance(IDirectory.class, domainUid).findByEntryUid(ownerUid);
			if (entry != null && entry.dataLocation != null) {
				loc = DataLocation.of(entry.dataLocation);
			}
		}
		IContainerStore containerStore = RepositoryProvider.instance(IContainerStore.class, context, loc);

		Container container = null;
		try {
			container = containerStore.get(containerUid);
		} catch (SQLException e) {
			throw new ServerFault(e);
		}

		if (container == null) {
			throw ServerFault.notFound("owner subs '" + containerUid + "' is missing");
		}
		OwnerSubscriptionsEventProducer events = new OwnerSubscriptionsEventProducer(domainUid, ownerUid,
				VertxPlatform.eventBus());
		return factoryClass().cast(new InternalOwnerSubscriptionsService(context, container, events));
	}

}
