/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2024
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/

package net.bluemind.core.container.persistence.factories;

import javax.sql.DataSource;

import net.bluemind.core.api.DataSourceType;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.persistence.DataSourceRouter;
import net.bluemind.core.container.persistence.ItemStore;
import net.bluemind.core.container.repository.IItemStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.repository.provider.IContainerBoundFactory;

public class ItemStoreFactory implements IContainerBoundFactory<IItemStore> {

	@Override
	public Class<IItemStore> factoryClass() {
		return IItemStore.class;
	}

	@Override
	public DataSourceType targetRepositoryType() {
		return DataSourceType.POSTGRESQL;
	}

	/**
	 * @param containerUid
	 * @return {@link IContainerManagement}
	 * @throws ServerFault if container not found or technical fault
	 */
	private IItemStore get(BmContext context, Container container) throws ServerFault {
		DataSource ds = DataSourceRouter.get(context, container.uid);
		return new ItemStore(ds, container, context.getSecurityContext());
	}

	@Override
	public IItemStore instance(BmContext context, Container container) throws ServerFault {
		if (container == null) {
			throw new ServerFault("container cannot be null");
		}
		return get(context, container);
	}

}
