/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.persistence;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;

public class AclCache {
	public static class Registration implements ICacheRegistration {

		@Override
		public void registerCaches(CacheRegistry cr) {
			var txCache = TxEnabler
					.wrap(Caffeine.newBuilder().recordStats().expireAfterWrite(10, TimeUnit.MINUTES).build());
			cr.register(AclCache.class, txCache);
		}
	}

	private final Cache<String, List<AccessControlEntry>> cache;

	public AclCache(Cache<String, List<AccessControlEntry>> c) {
		this.cache = c;
	}

	public static AclCache get(BmContext context) {
		if (context == null || context.provider().instance(CacheRegistry.class) == null) {
			return new AclCache(null);
		} else {
			return new AclCache(context.provider().instance(CacheRegistry.class).get(AclCache.class));
		}
	}

	public List<AccessControlEntry> getIfPresent(String uid) {
		if (cache != null) {
			return defensiveCopyNotNull(cache.getIfPresent(uid));
		} else {
			return null;
		}
	}

	public List<AccessControlEntry> get(String uid, Function<String, List<AccessControlEntry>> mappingFunction) {
		if (cache != null) {
			return defensiveCopyNotNull(cache.get(uid, mappingFunction));
		} else {
			return Collections.emptyList();
		}
	}

	private List<AccessControlEntry> defensiveCopyNotNull(List<AccessControlEntry> acls) {
		if (acls == null) {
			return Collections.emptyList();
		}
		return acls.stream().map(AccessControlEntry::copy).toList();
	}

	public void put(String uid, List<AccessControlEntry> c) {
		if (cache != null) {
			cache.put(uid, c);
		}
	}

	public void invalidate(String uid) {
		if (cache != null) {
			cache.invalidate(uid);
		}
	}

}
