/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.persistence;

import java.sql.SQLException;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;

public class DataSourceRouter {
	private DataSourceRouter() {
	}

	private static final Logger logger = LoggerFactory.getLogger(DataSourceRouter.class);
	private static final Cache<String, Optional<String>> cache = TxEnabler
			.wrap(Caffeine.newBuilder().recordStats().expireAfterAccess(2, TimeUnit.HOURS).build());

	public static class CacheReg implements ICacheRegistration {
		@Override
		public void registerCaches(CacheRegistry cr) {
			cr.register(DataSourceRouter.class, cache);
			logger.debug("Registered {}", cache);
		}
	}

	public static DataSource get(BmContext context, String containerUid) {
		Optional<String> loc = getImpl(context, containerUid);
		return loc.map(context::getMailboxDataSource).orElseGet(context::getDataSource);

	}

	private static Optional<String> getImpl(BmContext context, String containerUid) {
		Optional<String> emptyMeansDirectory = cache.get(containerUid, cUid -> {
			ContainerStore directoryContainerStore = new ContainerStore(context, context.getDataSource(),
					context.getSecurityContext());
			try {
				return directoryContainerStore.getContainerLocation(containerUid);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		});
		if (emptyMeansDirectory == null) { // NOSONAR
			return Optional.empty();
		}
		return emptyMeansDirectory;
	}

	public static String location(BmContext context, String containerUid) {
		Optional<String> loc = getImpl(context, containerUid);
		return loc.orElse(null);
	}

	public static void invalidateContainer(String containerUid) {
		cache.invalidate(containerUid);
	}

}
