/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.container.persistence;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ChangeLogEntry;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.repository.IChangelogStore;
import net.bluemind.core.container.repository.IWeightProvider;
import net.bluemind.core.jdbc.JdbcAbstractStore;

public class ChangelogStore extends JdbcAbstractStore implements IChangelogStore {

	@SuppressWarnings("unused")
	private static final Logger logger = LoggerFactory.getLogger(ChangelogStore.class);

	private final Container container;

	private class LightChangelogEntryPopulator implements EntityPopulator<ChangeLogEntry> {
		@Override
		public int populate(ResultSet rs, int index, ChangeLogEntry value) throws SQLException {
			value.version = rs.getLong(index++);
			value.type = ChangeLogEntry.Type.values()[rs.getByte(index++)];
			value.itemUid = rs.getString(index++);
			value.internalId = rs.getLong(index++);
			value.weightSeed = rs.getLong(index++);
			return index;
		}
	}

	private class FlaggedChangelogEntryPopulator implements EntityPopulator<FlaggedChangeLogEntry> {
		@Override
		public int populate(ResultSet rs, int index, FlaggedChangeLogEntry value) throws SQLException {
			value.version = rs.getLong(index++);
			value.type = ChangeLogEntry.Type.values()[rs.getByte(index++)];
			value.itemUid = rs.getString(index++);
			value.internalId = rs.getLong(index++);
			value.flags = ItemFlag.flags(rs.getInt(index++));
			value.weightSeed = rs.getLong(index++);
			return index;
		}
	}

	public ChangelogStore(DataSource pool, Container container) {
		super(pool);
		this.container = container;
	}

	// proc_insert_t_changeset(p_version,p_container_id,p_item_uid,p_type,p_date,p_item_id,p_weight_seed)
	private static final String INSERT_QUERY = "CALL proc_insert_t_changeset(?,?,?,?,?,?);";

	private static final String CHANGESET_QUERY = """
			    SELECT version, type, item_uid, item_id, weight_seed
			    FROM t_changeset
			    WHERE container_id = ? AND version >= ? order by item_id, version
			""";

	private static final String FLAGGED_CHANGESET_QUERY = """
			WITH changeset_filtered AS (
			       SELECT
			           version,
			           type,
			           item_uid,
			           item_id,
			           weight_seed,
			           container_id
			       FROM t_changeset
			       WHERE container_id = ?
			         AND version >= ?
			   ),
			   items_filtered AS (
			       SELECT
			           ci.container_id,
			           ci.id,
			           ci.version,
			           ci.flags
			       FROM t_container_item ci
			       WHERE ci.container_id = ?
			         AND EXISTS (
			             SELECT 1
			             FROM changeset_filtered cf
			             WHERE cf.item_id = ci.id
			               AND cf.version = ci.version
			         )
			   )
			   SELECT
			       cf.version,
			       cf.type,
			       cf.item_uid,
			       cf.item_id,
			       if.flags,
			       cf.weight_seed
			   FROM changeset_filtered cf
			   LEFT JOIN items_filtered if ON (
			       if.container_id = cf.container_id
			       AND if.id = cf.item_id
			       AND if.version = cf.version
			   )
			   ORDER BY cf.item_id, cf.version
					""";

	private static final String DELETE_CHANGESET_QUERY = "DELETE FROM t_changeset WHERE container_id = ?";

	private static final Creator<ChangeLogEntry> CREATOR = con -> new ChangeLogEntry();

	@Override
	public void itemCreated(LogEntry entry) throws SQLException {
		execute(INSERT_QUERY, new Object[] { entry.version, container.id, entry.itemUid, (byte) 0, entry.internalId,
				entry.weightSeed });
	}

	@Override
	public void itemUpdated(LogEntry entry) throws SQLException {
		execute(INSERT_QUERY, new Object[] { entry.version, container.id, entry.itemUid, (byte) 1, entry.internalId,
				entry.weightSeed });
	}

	@Override
	public void itemDeleted(LogEntry entry) throws SQLException {
		execute(INSERT_QUERY, new Object[] { entry.version, container.id, entry.itemUid, (byte) 2, entry.internalId,
				entry.weightSeed });
	}

	// FIXME: bm-4 Remove useless to parameter
	@Override
	public ContainerChangeset<String> changeset(long from, long to) throws SQLException {
		return changeset(s -> s, from, to);
	}

	@Override
	public ContainerChangeset<String> changeset(IWeightProvider wp, long from, long to) throws SQLException {
		List<ChangeLogEntry> entries = select(CHANGESET_QUERY, CREATOR, new LightChangelogEntryPopulator(),
				new Object[] { container.id, from });
		return ChangelogUtils.toChangeset(wp, from, entries, entry -> entry.itemUid, ItemFlagFilter.all());
	}

	// FIXME: bm-4 Remove useless to parameter
	@Override
	public ContainerChangeset<Long> changesetById(long from, long to) throws SQLException {
		return changesetById(s -> s, from, to);
	}

	@Override
	public ContainerChangeset<Long> changesetById(IWeightProvider wp, long from, long to) throws SQLException {
		List<ChangeLogEntry> entries = select(CHANGESET_QUERY, CREATOR, new LightChangelogEntryPopulator(),
				new Object[] { container.id, from });
		return ChangelogUtils.toChangeset(wp, from, entries, entry -> entry.internalId, ItemFlagFilter.all());
	}

	@Override
	public ContainerChangeset<ItemVersion> changesetById(long from, long to, ItemFlagFilter filter)
			throws SQLException {
		List<FlaggedChangeLogEntry> entries = select(FLAGGED_CHANGESET_QUERY, con -> new FlaggedChangeLogEntry(),
				new FlaggedChangelogEntryPopulator(), new Object[] { container.id, from, container.id });
		return ChangelogUtils.toChangeset(s -> s, from, entries, ItemVersion::new, filter);
	}

	@Override
	public ContainerChangeset<ItemVersion> changesetById(IWeightProvider wp, long from, long to, ItemFlagFilter filter)
			throws SQLException {
		List<FlaggedChangeLogEntry> entries = select(FLAGGED_CHANGESET_QUERY, con -> new FlaggedChangeLogEntry(),
				new FlaggedChangelogEntryPopulator(), new Object[] { container.id, from, container.id });
		return ChangelogUtils.toChangeset(wp, from, entries, ItemVersion::new, filter);
	}

	@Override
	public ContainerChangeset<ItemIdentifier> fullChangesetById(IWeightProvider wp, long from, long to)
			throws SQLException {
		List<FlaggedChangeLogEntry> entries = select(FLAGGED_CHANGESET_QUERY, con -> new FlaggedChangeLogEntry(),
				new FlaggedChangelogEntryPopulator(), new Object[] { container.id, from, container.id });
		return ChangelogUtils.toChangeset(wp, from, entries, ItemIdentifier::new, ItemFlagFilter.all());
	}

	@Override
	public void deleteLog() throws SQLException {
		delete(DELETE_CHANGESET_QUERY, new Object[] { container.id });
	}

	private static final String INSERT_ITEMS_DELETED_QUERY = "INSERT INTO t_changeset (version, container_id, item_uid, type, date, item_id) "
			+ " ( SELECT seq.seq +  row_number() over () , i.container_id, i.uid, 2, now(), i.id FROM t_container_item i, t_container_sequence seq WHERE i.container_id = ? AND seq.container_id = i.container_id)";

	@Override
	public void allItemsDeleted(String subject, String origin) throws SQLException {
		int insertCount = insert(INSERT_ITEMS_DELETED_QUERY, new Object[] { container.id });
		update("update t_container_sequence set seq = seq+? where container_id = ?",
				new Object[] { insertCount, container.id });
	}

}
