/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.container.persistence;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.sql.DataSource;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.i18n.labels.I18nLabels;

public class ContainerStore extends JdbcAbstractStore implements IContainerStore {

	// FIXME stores should not be aware of security context ?
	private SecurityContext securityContext;

	private ContainerCache cache;

	/**
	 * https://shipilev.net/jvm-anatomy-park/10-string-intern/
	 */
	public static class CHMInterner {
		private final Map<String, String> map;

		public CHMInterner() {
			map = new ConcurrentHashMap<>();
		}

		public String intern(String s) {
			if (s == null) {
				return s;
			}
			String exist = map.putIfAbsent(s, s);
			return (exist == null) ? s : exist;
		}
	}

	private static final CHMInterner interner = new CHMInterner();

	private static final EntityPopulator<Container> CONTAINER_POPULATOR = new EntityPopulator<Container>() {

		@Override
		public int populate(ResultSet rs, int index, Container value) throws SQLException {
			value.id = rs.getLong(index++);
			value.uid = rs.getString(index++);
			value.type = interner.intern(rs.getString(index++));
			value.name = rs.getString(index++);
			value.owner = rs.getString(index++);
			value.createdBy = rs.getString(index++);
			value.updatedBy = rs.getString(index++);
			value.created = Date.from(rs.getTimestamp(index++).toInstant());
			value.updated = Date.from(rs.getTimestamp(index++).toInstant());
			value.defaultContainer = rs.getBoolean(index++);
			value.domainUid = interner.intern(rs.getString(index++));
			value.readOnly = rs.getBoolean(index++);
			return index;
		}

	};

	public ContainerStore(BmContext ctx, DataSource dataSource, SecurityContext securityContext) {
		super(dataSource);
		this.securityContext = securityContext;
		this.cache = ContainerCache.get(ctx, dataSource);
	}

	private List<Container> findBy(ContainerQuery query) throws SQLException {
		List<Object> commonParams = new ArrayList<>();
		StringBuilder commonConditions = new StringBuilder();

		if (query.type != null) {
			commonConditions.append(" AND container_type = ?");
			commonParams.add(query.type);
		}
		if (query.readonly != null) {
			commonConditions.append(" AND readonly = ?");
			commonParams.add(query.readonly);
		}
		if (query.name != null) {
			commonConditions.append(" AND name = ?");
			commonParams.add(query.name);
		}

		// owner containers
		StringBuilder ownedQuery = new StringBuilder("""
				    SELECT id, uid, container_type, name, owner, createdby, updatedby,
				           created, updated, defaultContainer, domain_uid, readonly
				    FROM t_container
				    WHERE owner = ?
				""");
		ownedQuery.append(commonConditions);

		List<Object> ownedParams = new ArrayList<>();
		ownedParams.add(query.owner);
		ownedParams.addAll(commonParams);

		return select(ownedQuery.toString(), rs -> new Container(),
				Arrays.<EntityPopulator<Container>>asList(CONTAINER_POPULATOR), ownedParams.toArray());
	}

	@Override
	public List<Container> findByContainerQuery(ContainerQuery containerquery) throws SQLException {
		return findBy(containerquery);
	}

	@Override
	public List<Container> findByType(String containerType) throws SQLException {
		String selectQuery = "SELECT id, uid, container_type, name, owner, createdby, updatedby, created, updated, defaultContainer, domain_uid, readonly FROM t_container AS c "
				+ " WHERE c.container_type = ?";

		return select(selectQuery, (rs) -> new Container(),
				Arrays.<EntityPopulator<Container>>asList(CONTAINER_POPULATOR), new Object[] { containerType });

	}

	@Override
	public List<Container> findAccessiblesByType(final ContainerQuery query) throws SQLException {
		if (null != query.name) {
			Set<Container> containers = new HashSet<>();
			List<String> names = I18nLabels.getInstance().getMatchingKeys(query.name, securityContext.getLang());
			names.add(query.name);
			for (String queryName : names) {
				query.name = queryName;
				containers.addAll(findAccessiblesByTypeImpl(query));
				if (query.size > 0 && containers.size() >= query.size) {
					break;
				}
			}
			return new ArrayList<>(containers);
		} else {
			return findAccessiblesByTypeImpl(query);
		}
	}

	private List<Container> findAccessiblesByTypeImpl(final ContainerQuery query) throws SQLException {
		List<Container> allContainers = new ArrayList<>();
		Set<Long> containerIds = new HashSet<>();
		List<Object> commonParams = new ArrayList<>();
		StringBuilder commonConditions = new StringBuilder();

		if (query.type != null) {
			commonConditions.append(" AND container_type = ?");
			commonParams.add(query.type);
		}
		if (query.readonly != null) {
			commonConditions.append(" AND readonly = ?");
			commonParams.add(query.readonly);
		}
		if (query.name != null) {
			if (query.name.startsWith("$$")) {
				commonConditions.append(" AND name = ?");
				commonParams.add(query.name);
			} else {
				commonConditions.append(" AND (upper(name) like upper(?) AND name NOT LIKE '$$%')");
				commonParams.add("%" + query.name + "%");
			}
		}

		// owner containers
		StringBuilder ownedQuery = new StringBuilder("""
				    SELECT id, uid, container_type, name, owner, createdby, updatedby,
				           created, updated, defaultContainer, domain_uid, readonly
				    FROM t_container
				    WHERE owner = ?
				""");
		ownedQuery.append(commonConditions);

		List<Object> ownedParams = new ArrayList<>();
		ownedParams.add(securityContext.getSubject());
		ownedParams.addAll(commonParams);

		List<Container> ownedContainers = select(ownedQuery.toString(), rs -> new Container(),
				Arrays.<EntityPopulator<Container>>asList(CONTAINER_POPULATOR), ownedParams.toArray());

		for (Container container : ownedContainers) {
			if (containerIds.add(container.id)) {
				allContainers.add(container);
			}
		}

		// thru acl accessible containers
		StringBuilder accessibleQuery = new StringBuilder("""
				    SELECT DISTINCT c.id, c.uid, c.container_type, c.name, c.owner, c.createdby, c.updatedby,
				           c.created, c.updated, c.defaultContainer, c.domain_uid, c.readonly
				    FROM t_container c
				    JOIN t_container_acl acl ON c.id = acl.container_id
				    WHERE acl.subject = ANY(?)
				""");

		List<Object> accessibleParams = new ArrayList<>();
		List<String> subjects = new ArrayList<>(securityContext.getMemberOf());
		subjects.add(securityContext.getSubject());
		subjects.add(securityContext.getContainerUid()); // public share
		subjects.add("public");
		accessibleParams.add(subjects.toArray(new String[0]));

		if (query.verb != null && !query.verb.isEmpty()) {
			accessibleQuery.append(" AND acl.verb = ANY(?)");
			Set<Verb> verbs = EnumSet.noneOf(Verb.class);
			for (Verb v : query.verb) {
				verbs.addAll(v.parentHierarchy());
			}
			accessibleParams.add(verbs.stream().map(Enum::name).toArray(String[]::new));
		}

		accessibleQuery.append(commonConditions);
		accessibleParams.addAll(commonParams);

		List<Container> accessibleContainers = select(accessibleQuery.toString(), rs -> new Container(),
				Arrays.<EntityPopulator<Container>>asList(CONTAINER_POPULATOR), accessibleParams.toArray());

		for (Container container : accessibleContainers) {
			if (containerIds.add(container.id)) {
				allContainers.add(container);
			}
		}
		allContainers.sort(Comparator.comparing(c -> c.name.toUpperCase()));
		if (query.size > 0 && query.size < allContainers.size()) {
			return allContainers.subList(0, query.size);
		}

		return allContainers;
	}

	private static record InsertReturn(long id, Date created, Date updated, String createdby, String updatedby) {
	}

	@Override
	public Container create(Container container) throws SQLException {

		String insertQuery = "INSERT INTO t_container (uid, container_type, "
				+ "name, owner, createdby, updatedby, created, updated, defaultContainer, domain_uid, readonly) "
				+ " VALUES (?, ?, ?, ?, ?, ?, now(), now(), ?, ?, ?) RETURNING id, created, updated, createdby, updatedby";
		InsertReturn iret = insertAndReturn(insertQuery, container,
				Arrays.<StatementValues<Container>>asList((con, statement, index, rowIndex, value) -> {
					statement.setString(index++, value.uid);
					statement.setString(index++, value.type);
					statement.setString(index++, value.name);
					statement.setString(index++, value.owner);
					String principal = securityContext.getSubject();
					statement.setString(index++, principal);
					statement.setString(index++, principal);
					statement.setBoolean(index++, value.defaultContainer);
					statement.setString(index++, value.domainUid);
					statement.setBoolean(index++, value.readOnly);
					return index;
				}), rs -> new InsertReturn(rs.getLong(1), Date.from(rs.getTimestamp(2).toInstant()),
						Date.from(rs.getTimestamp(3).toInstant()), rs.getString(4), rs.getString(5)),
				null);
		container.id = iret.id;
		container.created = iret.created;
		container.updated = iret.updated;
		container.createdBy = iret.createdby;
		container.updatedBy = iret.updatedby;
		cache.put(container.uid, container.id, container);
		// container settings
		insert("INSERT INTO t_container_settings (container_id, settings) VALUES (?, '')",
				new Object[] { container.id });
		// insert seq
		insert("INSERT INTO t_container_sequence (container_id) VALUES (?)", new Object[] { container.id });
		return container;
	}

	@Override
	public void update(final String uid, final String name, boolean defaultContainer) throws SQLException {
		String updateQuery = "UPDATE t_container SET (name, defaultcontainer, updatedby, updated) = (?, ?, ?, now()) WHERE uid = ?";
		update(updateQuery, null, (con, statement, index, currentRow, value) -> {
			statement.setString(index++, name);
			statement.setBoolean(index++, defaultContainer);
			statement.setString(index++, securityContext.getSubject());
			statement.setString(index++, uid);
			return index;
		});
		invalidateCache(uid, get(uid).id);
	}

	@Override
	public Container get(String uid) throws SQLException {
		Container c = cache.getIfPresent(uid);
		if (c == null) {
			String selectQuery = "SELECT id, uid, container_type, name, owner, createdby, updatedby, created, updated, defaultContainer, domain_uid, readonly FROM t_container WHERE uid = ?";

			c = unique(selectQuery, rs -> new Container(),
					Arrays.<EntityPopulator<Container>>asList(CONTAINER_POPULATOR), new Object[] { uid });
			if (c != null) {
				cache.put(uid, c.id, c);
				return c;
			} else {
				return null;
			}
		} else {
			return c;
		}
	}

	private Long getInternalId(String uid) throws SQLException {
		Container c = cache.getIfPresent(uid);
		if (c == null) {
			return unique("SELECT id FROM t_container WHERE uid = ?", rs -> rs.getLong(1), (rs, index, value) -> index,
					new Object[] { uid });
		} else {
			return c.id;
		}
	}

	@Override
	public Container get(long id) throws SQLException {
		Container c = cache.getIfPresent(id);
		if (c == null) {
			String selectQuery = "SELECT id, uid, container_type, name, owner, createdby, updatedby, created, updated, defaultContainer, domain_uid, readonly FROM t_container WHERE id = ?";

			c = unique(selectQuery, rs -> new Container(),
					Arrays.<EntityPopulator<Container>>asList(CONTAINER_POPULATOR), new Object[] { id });
			if (c != null) {
				cache.put(c.uid, c.id, c);
				return c;
			} else {
				return null;
			}
		} else {
			return c;
		}
	}

	@Override
	public boolean exists(String uid) throws SQLException {
		return getInternalId(uid) != null;
	}

	public void deleteAllSubscriptions(Container container) throws SQLException {
		delete("DELETE FROM t_container_sub WHERE container_uid = ? ", new Object[] { container.uid });
	}

	@Override
	public void delete(String uid) throws SQLException {
		Long internalId = getInternalId(uid);
		if (internalId == null) {
			throw ServerFault.notFound(uid);
		}
		deleteKnownIdUid(internalId, uid);
	}

	@Override
	public void delete(Container existing) throws SQLException {
		deleteKnownIdUid(existing.id, existing.uid);
	}

	private void deleteKnownIdUid(long id, String uid) throws SQLException {
		delete("DELETE FROM t_container_settings WHERE container_id = ?", new Object[] { id });
		delete("DELETE FROM t_container_sequence WHERE container_id = ?", new Object[] { id });
		delete("DELETE FROM t_container_item WHERE container_id = ?", new Object[] { id });
		delete("DELETE FROM t_container WHERE id = ?", new Object[] { id });
		invalidateCache(uid, id);
	}

	public void invalidateCache(String uid, Long id) {
		cache.invalidate(uid, id);
	}

	public List<String> listSubscriptions(Container container) throws SQLException {
		return select(
				"SELECT uid FROM t_container_sub INNER JOIN t_container_item ON id=user_id WHERE t_container_sub.container_uid = ?",
				StringCreator.FIRST, Collections.emptyList(), new Object[] { container.uid });

	}

	/**
	 * Creates or updates given container's location
	 *
	 * @param container
	 * @param location
	 * @throws SQLException
	 */
	public void createOrUpdateContainerLocation(Container container, String location) throws SQLException {
		createOrUpdateContainerLocation(container.uid, location);
	}

	public void createOrUpdateContainerLocation(String contUid, String location) throws SQLException {
		insert("INSERT INTO t_container_location VALUES (?, ?) ON CONFLICT (container_uid) DO UPDATE SET location = ? WHERE t_container_location.container_uid = ?",
				new Object[] { contUid, location, location, contUid });
		DataSourceRouter.invalidateContainer(contUid);
	}

	public void deleteContainerLocation(Container container) throws SQLException {
		deleteContainerLocation(container.uid);
	}

	public void deleteContainerLocation(String containerUid) throws SQLException {
		delete("DELETE FROM t_container_location WHERE container_uid = ?", new Object[] { containerUid });
		DataSourceRouter.invalidateContainer(containerUid);
	}

	/**
	 * Returns null if the container location is unknown, or an optional if the
	 * location is known.
	 *
	 * @param containerUid
	 * @return
	 * @throws SQLException
	 */
	public Optional<String> getContainerLocation(String containerUid) throws SQLException {
		String ret = unique("SELECT coalesce(location, 'DIR') FROM t_container_location WHERE container_uid = ?",
				StringCreator.FIRST, Collections.emptyList(), new Object[] { containerUid });
		if (ret == null) {
			return null;
		} else if ("DIR".equals(ret)) {
			return Optional.empty();
		} else {
			return Optional.of(ret);
		}
	}

	/**
	 * Returns all container uids belonging to a different server
	 *
	 * @param serverUid
	 * @return
	 * @throws SQLException
	 */
	private List<String> getForeignContainers(String location) throws SQLException {
		if (location == null) {
			return select(
					"SELECT container_uid FROM t_container_location WHERE location IS NOT NULL AND location <> ''",
					StringCreator.FIRST, Collections.emptyList(), new Object[0]);
		} else {
			return select(
					"SELECT c.uid FROM t_container c LEFT JOIN t_container_location cl ON c.uid = cl.container_uid WHERE cl.location IS NULL OR cl.location = '' OR cl.location <> ?",
					StringCreator.FIRST, Collections.emptyList(), new Object[] { location });
		}
	}

	public Set<String> getObsoleteContainers(String location) throws SQLException {
		List<String> containers = select("SELECT c.uid FROM t_container c WHERE c.container_type = ?",
				StringCreator.FIRST, Collections.emptyList(), new Object[] { "t_folder" });
		containers.addAll(getForeignContainers(location));
		return new HashSet<>(containers);
	}

	public Set<String> getMissingContainerSequence() throws SQLException {
		String query = "select uid from t_container where id not in (select container_id from t_container_sequence)";
		List<String> containers = select(query, StringCreator.FIRST, Collections.emptyList(), new Object[] {});
		return new HashSet<>(containers);
	}

	public Set<String> getMissingContainerSettings() throws SQLException {
		String query = "select uid from t_container where id not in (select container_id from t_container_settings)";
		List<String> containers = select(query, StringCreator.FIRST, Collections.emptyList(), new Object[] {});
		return new HashSet<>(containers);
	}

	public void createMissingContainerSequence() throws SQLException {
		insert("insert into t_container_sequence select container_id, max(version) from t_container_item where container_id not in (select container_id from t_container_sequence) GROUP BY container_id",
				new Object[] {});
		insert("insert into t_container_sequence select id, 0 from t_container where id not in (select container_id from t_container_sequence)",
				new Object[] {});
	}

	public void createMissingContainerSettings() throws SQLException {
		insert("insert into t_container_settings select id, '' from t_container where id not in(select container_id from t_container_settings)",
				new Object[] {});
	}
}
