/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.utils;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Policy;
import com.github.benmanes.caffeine.cache.stats.CacheStats;

/**
 * A cache implementation that remains an empty collection and returns null when
 * trying to access records, only the get function
 * {@code V get(K key, Function<? super K, ? extends V> mappingFunction)} will
 * execute the mapping function if it exists
 * 
 * @param <K>
 * @param <V>
 */
public class NoopCache<K, V> implements Cache<K, V> {

	@Override
	public V getIfPresent(Object key) {
		return null;
	}

	@Override
	public V get(K key, Function<? super K, ? extends V> mappingFunction) {
		return mappingFunction != null ? mappingFunction.apply(key) : null;
	}

	@Override
	public Map<K, V> getAllPresent(Iterable<? extends K> keys) {
		return Collections.emptyMap();
	}

	@Override
	public void put(K key, V value) {
	}

	@Override
	public void putAll(Map<? extends K, ? extends V> map) {
	}

	@Override
	public void invalidate(K key) {
	}

	@Override
	public void invalidateAll(Iterable<? extends K> keys) {
	}

	@Override
	public void invalidateAll() {
	}

	@Override
	public long estimatedSize() {
		return 0;
	}

	@Override
	public CacheStats stats() {
		return null;
	}

	@Override
	public ConcurrentMap<K, V> asMap() {
		return new ConcurrentHashMap<>();
	}

	@Override
	public void cleanUp() {

	}

	@Override
	public Policy<K, V> policy() {
		return null;
	}

	@Override
	public Map<K, V> getAll(Iterable<? extends K> keys,
			Function<? super Set<? extends K>, ? extends Map<? extends K, ? extends V>> mappingFunction) {
		return Collections.emptyMap();
	}

}
