/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.backup.continuous.index;

import java.io.DataOutput;
import java.io.IOException;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.Unpooled;
import io.vertx.core.buffer.impl.BufferImpl;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.backup.continuous.model.RecordKey;

public interface ISerde<T> {
	T deserialize(byte[] data);

	byte[] serialize(T value);

	public static final ISerde<String> UTF8_STRING = new ISerde<>() {

		@Override
		public String deserialize(byte[] data) {
			return data == null ? null : new String(data);
		}

		@Override
		public byte[] serialize(String value) {
			return value == null ? null : value.getBytes();
		}
	};

	public static final ISerde<JsonObject> JSON = new ISerde<JsonObject>() {

		@Override
		public JsonObject deserialize(byte[] data) {
			return data == null || data.length == 0 ? null : new JsonObject(BufferImpl.buffer(data));
		}

		@Override
		public byte[] serialize(JsonObject value) {
			return value == null ? null : value.encode().getBytes();
		}
	};

	public static final ISerde<byte[]> PRISTINE = new ISerde<byte[]>() {

		@Override
		public byte[] deserialize(byte[] data) {
			return data;
		}

		@Override
		public byte[] serialize(byte[] value) {
			return value;
		}
	};

	public static ISerde<String> jsonStringField(String field) {

		return new ISerde<String>() {
			JsonFactory jsonFactory = new JsonFactory();

			@Override
			public String deserialize(byte[] data) {
				if (data == null) {
					return null;
				}
				try (JsonParser parser = jsonFactory.createParser(data)) {

					int depth = 0;

					while (parser.nextToken() != null) {
						JsonToken token = parser.getCurrentToken();

						if (token == JsonToken.START_OBJECT || token == JsonToken.START_ARRAY) {
							depth++;
						} else if (token == JsonToken.END_OBJECT || token == JsonToken.END_ARRAY) {
							depth--;
						}

						if (depth == 1 && token == JsonToken.FIELD_NAME && field.equals(parser.currentName())) {
							parser.nextToken(); // Move to value
							return parser.getValueAsString();
						}
					}
					return null;
				} catch (Exception e) {
					throw new ServerFault(e);
				}
			}

			@Override
			public byte[] serialize(String value) {
				return new JsonObject().put(field, value).toBuffer().getBytes();
			}

		};
	}

	public static final ISerde<RecordKey> RECORD_KEY = new ISerde<RecordKey>() {

		private final JsonFactory jsonFactory = new JsonFactory();

		@Override
		public RecordKey deserialize(byte[] data) {
			if (data == null) {
				return null;
			}
			try (JsonParser parser = jsonFactory.createParser(data)) {
				if (parser.nextToken() != JsonToken.START_OBJECT) {
					throw new ServerFault("Not a RecordKey object start");
				}
				return deserialize0(parser);
			} catch (IOException e) {
				throw new ServerFault(e);
			}
		}

		private final RecordKey deserialize0(JsonParser parser) throws IOException {
			RecordKey rk = new RecordKey();
			JsonToken tok;
			while ((tok = parser.nextToken()) != JsonToken.END_OBJECT && tok != null) {
				String fieldName = parser.currentName();
				parser.nextToken();
				switch (fieldName) {
				case "type" -> rk.type = parser.getText();
				case "owner" -> rk.owner = parser.getText();
				case "uid" -> rk.uid = parser.getText();
				case "id" -> rk.id = parser.getLongValue();
				case "valueClass" -> rk.valueClass = parser.getText();
				case "operation" -> rk.operation = parser.getText();
				default -> parser.skipChildren();
				}
			}
			return rk;
		}

		@Override
		public byte[] serialize(RecordKey rk) {
			byte[] ret = new byte[2048];
			ByteBuf unpooled = Unpooled.wrappedBuffer(ret).readerIndex(0).writerIndex(0);
			ByteBufOutputStream out = new ByteBufOutputStream(unpooled, false);
			try (JsonGenerator writer = jsonFactory.createGenerator((DataOutput) out)) {
				writer.writeStartObject();
				writer.writeStringField("type", rk.type);
				writer.writeStringField("owner", rk.owner);
				writer.writeNumberField("id", rk.id);
				writer.writeStringField("valueClass", rk.valueClass);
				writer.writeStringField("operation", rk.operation);
				writer.writeEndObject();
			} catch (Exception e) {
				throw new ServerFault(e);
			}
			byte[] justData = new byte[unpooled.readableBytes()];
			System.arraycopy(ret, 0, justData, 0, justData.length);
			return justData;
		}

	};
}
