/*
 * BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
goog.provide("net.bluemind.contact.vcard.edit.ui.MultipleValueField");

goog.require("goog.events.FocusHandler");
goog.require("goog.events.InputHandler");
goog.require("goog.events.FocusHandler.EventType");
goog.require("goog.events.InputHandler.EventType");
goog.require("goog.ui.Button");
goog.require("goog.ui.Control");
goog.require("goog.ui.LabelInput");
goog.require("goog.ui.Component.EventType");
goog.require("goog.ui.Component.State");
goog.require("net.bluemind.ui.form.FormField");
goog.require("net.bluemind.ui.form.templates");
goog.require("bluemind.ui.style.TrashButtonRenderer");

/**
 * @constructor
 * 
 * @param {goog.ui.ControlContent} label
 * @param {*} options
 * @param {goog.ui.ControlRenderer=} opt_renderer
 * @param {goog.dom.DomHelper=} opt_domHelper
 * @extends {net.bluemind.ui.form.FormField}
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField = function(label, opt_renderer, opt_domHelper) {
  net.bluemind.ui.form.FormField.call(this, label, opt_renderer, opt_domHelper);
  this.addClassName(goog.getCssName('field-multi'));

}
goog.inherits(net.bluemind.contact.vcard.edit.ui.MultipleValueField, net.bluemind.ui.form.FormField);


/** @override */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.createFormField = function() {
  // Add container
  var container = new goog.ui.Control();
  container.setHandleMouseEvents(false);
  container.setSupportedState(goog.ui.Component.State.FOCUSED, false);
  container.addClassName(goog.getCssName('field-base'));
  this.addChild(container, true);

  // Add trash to previous child
  var previous = this.getChildAt(this.indexOfChild(container) - 1);
  if (previous && previous.getChild('field') && !previous.getChild('trash')) {
    this.addTrashIcon(previous);
  }


  container.getElement().innerHTML = net.bluemind.ui.form.templates.field();



  this.createLabel(container);

  var field = this.createField();
  container.addChild(field);
  field.render(container.getElementByClass(goog.getCssName('field-base-field')));
  field.setEnabled(this.isEnabled());

  var input = new goog.events.InputHandler(field.getElement());
  container.registerDisposable(input);
  var focus = new goog.events.FocusHandler(field.getElement())
  container.registerDisposable(focus);


  this.getHandler().listen(input, goog.events.InputHandler.EventType.INPUT, function(e) {
    if (this.indexOfChild(container) == (this.getChildCount() - 1) && field.getValue() != '') {
      this.createFormField();
    }
    this.handleInputChanged(container);
  });

  this.getHandler().listen(focus, goog.events.FocusHandler.EventType.FOCUSOUT, function() {
    if (this.indexOfChild(container) != (this.getChildCount() - 1) && field.getValue() == '') {
      this.removeFormField(container);
    }
  });

  return container;
};

net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.addTrashIcon = function(container) {
    var trash = new goog.ui.Button(" ", bluemind.ui.style.TrashButtonRenderer.getInstance());
    container.addChild(trash);
    trash.render(container.getElementByClass(goog.getCssName('field-base-field')));
    this.getHandler().listen(trash, goog.ui.Component.EventType.ACTION, function() {
        this.removeFormField(container);
    });
}


/**
 * Remove a row
 * 
 * @param {goog.ui.Control} container
 * @protected
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.removeFormField = function(container) {
  this.removeChild(container, true);
  container.dispose();
};


/** @override */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.createField = function() {
  var field = this.createInput();
  field.setId('field');
  return field;
};

net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.isEnabled = function() {
  return true;
};

/**
 * Create input field
 * 
 * @return {goog.ui.LabelInput}
 * @protected
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.createInput = function() {
  return new goog.ui.LabelInput(this.label);
};

/**
 * Return field css
 * 
 * @return {Array}
 * @protected
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.getFieldClassNames = function() {
  return [ goog.getCssName('field-text') ];
};

/**
 * Reset value
 * 
 * @protected
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.resetValue = function() {
  goog.array.forEach(this.removeChildren(true), function(child) {
    child.dispose();
  })
  this.createFormField();
};

/** @override */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.setValue = function(values) {
  this.resetValue();
  values = values || [];
  goog.array.forEach(values, function(value) {
    var child = this.getChildAt(this.getChildCount() - 1);
    this.setFieldValue(child, value);
    this.createFormField();
  }, this);
};

/**
 * @param {goog.ui.Component} component
 * @param {*} value
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.setFieldValue = function(component, value) {
  component.getChild('field').setValue(value);
};


/** @override */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.getValue = function() {
  var value = [];
  this.forEachChild(function(child) {
    if (child.getChild('field')) {
      var result = this.getFieldValue(child);
      if (result) {
        value.push(result);

      }
    }
  }, this);
  return value;
};

/**
 * @param {goog.ui.Component} component
 * @param {*} value
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.getFieldValue = function(component) {
  return component.getChild('field').getValue();
};


/**
 * @protected
 * @param {goog.ui.Control} container
 */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.handleInputChanged = function(container) {
};

/** @override */
net.bluemind.contact.vcard.edit.ui.MultipleValueField.prototype.createLabel = function(container) {
    var el = container.getElementByClass(goog.getCssName('field-base-label'));
    if (el && this.label) {
      if (goog.isString(this.label)) {
        this.getDomHelper().setTextContent(el, this.label);
      } else {
        var childHandler = function(child) {
          if (child) {
            var doc = goog.dom.getOwnerDocument(el);
            el.appendChild(goog.isString(child) ? doc.createTextNode(child) : child);
          }
        };
        if (goog.isArray(this.label)) {
          goog.array.forEach(this.label, childHandler);
        } else if (goog.isArrayLike(this.label) && !('nodeType' in this.label)) {
          goog.array.forEach(goog.array.clone((this.label)), childHandler);
        } else {
          childHandler(this.label);
        }
      }
    }
  };