/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2019
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.sysconf;

import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.stream.Collectors;

import io.vertx.core.json.JsonObject;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.system.api.IInternalCredentials;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "list", description = "Show Sysconf values.")
public class SysconfListCommand implements ICmdLet, Runnable {

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("sysconf");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return SysconfListCommand.class;
		}
	}

	protected CliContext ctx;
	protected CliUtils cliUtils;

	@Option(names = { "--json", "-j" }, required = false, description = "Format list as json")
	public boolean displayAsJson = false;

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}

	@Override
	public void run() {
		SystemConf sysConf = retrievePlainSystemConf();

		if (displayAsJson) {
			displayAsJson(sysConf.values);
		} else {
			displayAsArray(sysConf.values);
		}
	}

	private SystemConf retrievePlainSystemConf() {
		ISystemConfiguration configurationApi = ctx.adminApi().instance(ISystemConfiguration.class);

		SystemConf sysconf = configurationApi.getValues();

		return SystemConf.create(SystemConf.merge(sysconf,
				sysconf.values.entrySet().stream().filter(e -> SysConfKeys.obfuscateOnGet(e.getKey()))
						.map(this::getPlainValue).collect(Collectors.toMap(Entry::getKey, Entry::getValue))));
	}

	private Entry<String, String> getPlainValue(Entry<String, String> entry) {
		IInternalCredentials credentials = ctx.adminApi().instance(IInternalCredentials.class, "global.virt");
		entry.setValue(credentials.getDomainCredentialById(entry.getKey()));

		return entry;
	}

	private void displayAsArray(Map<String, String> map) {
		String[] headers = { "Attribute", "Value" };
		ctx.info(cliUtils.display(map, headers));
	}

	private void displayAsJson(Map<String, String> map) {
		JsonObject jsonObject = new JsonObject();
		map.forEach((k, v) -> jsonObject.put(k, v));
		ctx.info(jsonObject.encode());
	}

}