/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.mailshare;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.slf4j.helpers.MessageFormatter;

import io.netty.util.internal.ThreadLocalRandom;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCard.Communications.Tel;
import net.bluemind.addressbook.api.VCard.Identification.FormatedName;
import net.bluemind.addressbook.api.VCard.Identification.Name;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.core.api.Email;
import net.bluemind.core.email.EmailHelper;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.mailbox.api.Mailbox.Routing;
import net.bluemind.mailshare.api.IInternalMailshare;
import net.bluemind.mailshare.api.Mailshare;
import net.datafaker.Faker;
import net.datafaker.providers.base.PhoneNumber;
import net.datafaker.providers.entertainment.GameOfThrones;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

@Command(name = "quickcreate", description = "Create a mail-enabled mailshare")
public class MailshareQuickCreateCommand implements ICmdLet, Runnable {

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("mailshare");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return MailshareQuickCreateCommand.class;
		}
	}

	@Parameters(paramLabel = "<login@domain>", description = "login@domain of the mailshare to create (if --count is specified, {} must be present in the login for the counter)")
	public String loginAtDomain;

	@Option(names = "--random", description = "Generate random infos into the VCard")
	public Boolean randomData = false;

	@Option(names = "--workers", description = "Number of threads to create mailshares")
	public int workers = 4;

	@Option(names = "--count", description = "Number of mailshares to create (default=1)")
	public int count = 1;

	private CliContext ctx;

	private static final net.datafaker.providers.base.Name nameFaker = new Faker().name();
	private static final PhoneNumber phoneFaker = new Faker().phoneNumber();
	private static final GameOfThrones gotFaker = new Faker().gameOfThrones();

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	private String forEmail(String namePart) {
		return namePart.replace(' ', '.').toLowerCase();
	}

	public void run() {
		if (!EmailHelper.isValid(loginAtDomain)) {
			ctx.error(loginAtDomain + " does not look like a valid email.");
			return;
		}
		MQ.init().join();

		String[] splitted = loginAtDomain.split("@");
		String localPart = splitted[0];
		String domainName = splitted[1];

		CliUtils cliUtils = new CliUtils(ctx);
		String domainUid = cliUtils.getDomainUidByDomain(domainName);

		if (count == 1) {
			createMailshare(localPart, domainName, domainUid);
		} else {
			if (!localPart.contains("{}")) {
				ctx.error("NO: login@domain must contain '{}' for the counter");
			} else {
				ArrayBlockingQueue<String> q = new ArrayBlockingQueue<>(workers);
				try (ExecutorService pool = Executors.newFixedThreadPool(workers)) {
					for (int i = 0; i < count; i++) {
						String login = MessageFormatter.format(localPart, i).getMessage();
						try {
							q.put(login);
						} catch (InterruptedException ie) {
							Thread.currentThread().interrupt();
						}
						pool.submit(() -> {
							try {
								createMailshare(login, domainName, domainUid);
							} finally {
								q.remove(login); // NOSONAR
							}
						});
					}
				}
			}
		}

	}

	private void createMailshare(String login, String domainName, String domainUid) {
		String latd = login + "@" + domainName;
		Mailshare ms = new Mailshare();
		ms.name = login;
		VCard card = new VCard();
		String familyName = domainName.toUpperCase();
		String givenName = login;
		String extraName = null;
		String fn = givenName + " " + familyName;
		List<Email> emails = new ArrayList<>();
		Email defEmail = Email.create(latd, true, false);
		emails.add(defEmail);
		if (Boolean.TRUE.equals(randomData)) {
			familyName = nameFaker.lastName();
			givenName = nameFaker.firstName();
			extraName = login;
			fn = givenName + " " + extraName + " " + familyName;
			card.organizational.role = gotFaker.house();
			card.communications.tels = Arrays.asList(Tel.create(phoneFaker.cellPhone(), Collections.emptyList()));

			String fakeLocal = forEmail(givenName) + "." + forEmail(familyName)
					+ ThreadLocalRandom.current().nextInt(100, 1000);
			defEmail.isDefault = false;
			emails.add(Email.create(fakeLocal + "@" + domainName, true, false));
		}
		card.identification.name = Name.create(familyName, givenName, extraName, null, null, null);
		card.identification.formatedName = FormatedName.create(fn);
		ms.card = card;

		ms.routing = Routing.internal;
		ms.emails = emails;

		IInternalMailshare mailshareApi = ctx.adminApi().instance(IInternalMailshare.class, domainUid);
		String uid = "cli-created-" + UUID.nameUUIDFromBytes(latd.getBytes()).toString().toLowerCase();
		try {
			mailshareApi.createMailshare(uid, ms);
			ctx.info("Created {} for {}", uid, latd);
		} catch (Exception e) {
			ctx.error("Error on {} creation for {}: {}", uid, latd, e.getMessage());
		}
	}

}
