/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.launcher;

import java.io.PrintWriter;
import java.util.Map;

import picocli.CommandLine;
import picocli.CommandLine.Command;
import picocli.CommandLine.Help;
import picocli.CommandLine.IHelpCommandInitializable2;
import picocli.CommandLine.Option;
import picocli.CommandLine.ParameterException;
import picocli.CommandLine.Parameters;

@Command(name = "help", header = "Display help information about the specified command.", synopsisHeading = "%nUsage: ", helpCommand = true, description = {
		"%nWhen no COMMAND is given, the usage help for the main command is displayed.",
		"If a COMMAND is specified, the help for that command is shown.%n" })
public class HelpCommand implements IHelpCommandInitializable2, Runnable {

	@Option(names = { "-h",
			"--help" }, usageHelp = true, descriptionKey = "helpCommand.help", description = "Show usage help for the help command and exit.")
	private boolean helpRequested;

	@Parameters(paramLabel = "COMMAND", arity = "0..1", descriptionKey = "helpCommand.command", description = "The COMMAND to display the usage help message for.")
	private String commands;

	private CommandLine self;
	private PrintWriter outWriter;
	private Help.Ansi ansi; // for backwards compatibility with pre-4.0
	private Help.ColorScheme colorScheme;

	public void run() {
		CommandLine parent = self == null ? null : self.getParent();
		if (parent == null) {
			return;
		}
		Help.ColorScheme colors = colorScheme != null ? colorScheme : Help.defaultColorScheme(ansi);
		if (commands != null) {
			Map<String, CommandLine> parentSubcommands = parent.getCommandSpec().subcommands();
			CommandLine subcommand = parentSubcommands.get(commands);

			if (subcommand != null) {
				subcommand.usage(outWriter, colors);
			} else {
				throw new ParameterException(parent, "Unknown subcommand '" + commands + "'.", null, commands);
			}
		} else {
			parent.usage(outWriter, colors);
		}
	}

	public void init(CommandLine helpCommandLine, Help.ColorScheme colorScheme, PrintWriter out, PrintWriter err) {
		this.self = helpCommandLine;
		this.colorScheme = colorScheme;
		this.outWriter = out;
	}

}
