/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.authentication;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.Tasks;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.api.TaskStatus;
import net.bluemind.keycloak.api.IKeycloakAdmin;
import picocli.CommandLine.Command;

@Command(name = "reconfigure", description = "Delete & recreate keycloak realms for existing domains")
public class AuthReconfigureCommand implements ICmdLet, Runnable {
	public static class Reg implements ICmdLetRegistration {
		@Override
		public Optional<String> group() {
			return Optional.of("auth");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return AuthReconfigureCommand.class;
		}
	}

	private CliContext ctx;

	@Override
	public void run() {
		IKeycloakAdmin kcAdm = ctx.adminApi().instance(IKeycloakAdmin.class);
		TaskRef tr = kcAdm.reconfigure();
		CompletableFuture<TaskStatus> done = Tasks.followStream(ctx, "auth-reconf", tr, true);
		TaskStatus status = done.orTimeout(1, TimeUnit.MINUTES).join();
		if (status.state.succeed) {
			ctx.info("Reconfiguration completed successfully");
		} else {
			ctx.warn("Reconfiguration failed: {}", status.lastLogEntry);
		}
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}
}
