/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.cli.auditlog;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.core.auditlogs.AuditLogEntry;
import net.bluemind.core.auditlogs.AuditLogQuery;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.domain.api.Domain;
import picocli.CommandLine.ArgGroup;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "mail", description = "get mail data")
public class GetMailAuditLogCommand extends AbstractGetAuditLogCommand {

	private static final String[] HEADERS = { "Timestamp", "Action", "Action responsible", "Mail folder name",
			"Mail folder owner", "Mail folder uid", "Mail Subject", "Message Id", "senders", "recipients",
			"information", "log type" };

	private static final String[] LOG_TYPE = { "mailbox_records", "enginerule" };

	@ArgGroup(exclusive = true)
	MessageQueryOption messageQueryOption;

	@Option(names = "--folderName", required = false, description = "the folder name to query")
	public String folderName;

	private static class MessageQueryOption {
		@Option(names = "--messageId", required = true, description = "the message id to query")
		public String messageId;

		@Option(names = "--subject", required = true, description = "the subject to query")
		public String subject;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("auditlog");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return GetMailAuditLogCommand.class;
		}

	}

	@Override
	protected String[][] generateOutputTable(List<AuditLogEntry> logMailQueryResults) {
		String[][] data = new String[logMailQueryResults.size()][HEADERS.length];
		int index = 0;
		for (AuditLogEntry log : logMailQueryResults) {
			data[index][0] = log.timestamp.toString();
			data[index][1] = log.action;
			data[index][2] = (log.securityContext != null) ? log.securityContext.email() : "";
			data[index][3] = (log.container != null) ? log.container.name() : "";
			data[index][4] = (log.container != null) ? log.container.ownerElement().email() : "";
			data[index][5] = (log.container != null) ? log.container.uid() : "";
			data[index][6] = (log.content != null) ? log.content.description() : "";
			data[index][7] = (log.content != null) ? log.content.key() : "";
			data[index][8] = (log.content != null)
					? log.content.author().stream().reduce((p, e) -> p + ", " + e).orElse("")
					: "";
			if (log.content != null) {
				List<String> receivers = new ArrayList<String>(log.content.with());
				receivers.removeAll(log.content.author());
				data[index][9] = receivers.stream().reduce((p, e) -> p + ", " + e).orElse("");
			}
			data[index][10] = log.updatemessage;
			data[index][11] = log.logtype;
			index++;
		}
		return data;
	}

	@Override
	protected AuditLogQuery buildQuery(ItemValue<Domain> domain) {
		AuditLogQuery auditLogQuery = defaultQuery();

		if (folderName != null && !folderName.isBlank()) {
			auditLogQuery.containerName = folderName;
		}

		if (messageQueryOption != null) {
			if (messageQueryOption.subject != null && !messageQueryOption.subject.isBlank()) {
				auditLogQuery.description = messageQueryOption.subject;
			}
			if (messageQueryOption.messageId != null && !messageQueryOption.messageId.isBlank()) {
				auditLogQuery.key = messageQueryOption.messageId;
			}
		}

		return auditLogQuery;
	}

	@Override
	protected String[] headers() {
		return HEADERS;
	}

	@Override
	protected String[] logTypes() {
		return LOG_TYPE;
	}

	@Override
	protected Map<String, String> generateOutputJSON(AuditLogEntry result) {
		Map<String, String> map = new HashMap<>();
		map.put("timestamp", result.timestamp.toString());
		map.put("action", result.action);
		map.put("securitycontext_owner", (result.securityContext != null) ? result.securityContext.email() : "");
		map.put("container_name", (result.container != null) ? result.container.name() : "");
		map.put("container_owner", (result.container != null) ? result.container.ownerElement().email() : "");
		map.put("container_uid", (result.container != null) ? result.container.uid() : "");
		map.put("content_description", (result.content != null) ? result.content.description() : "");
		map.put("content_key", (result.content != null) ? result.content.key() : "");
		map.put("content_author",
				(result.content != null) ? result.content.author().stream().reduce((p, e) -> p + ", " + e).orElse("")
						: "");
		if (result.content != null) {
			List<String> receivers = new ArrayList<String>(result.content.with());
			receivers.removeAll(result.content.author());
			map.put("content_recipients", receivers.stream().reduce((p, e) -> p + ", " + e).orElse(""));
		}
		map.put("updatemessage", result.updatemessage);
		map.put("logtype", result.logtype);

		return map;
	}
}
