/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.cli.auditlog;

import java.util.Date;
import java.util.List;
import java.util.Optional;

import com.github.freva.asciitable.AsciiTable;

import io.netty.util.internal.StringUtil;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.core.auditlogs.AuditLogEntry;
import net.bluemind.core.auditlogs.AuditLogQuery;
import net.bluemind.core.auditlogs.api.ILogRequestService;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import picocli.CommandLine.ArgGroup;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "global", description = "get global auditlog data")
public class GetGLobalAuditLogCommand implements Runnable, ICmdLet {

	enum OutputFormat {
		json, table
	}

	enum LogType {
		calendar, mailbox_records, login, containeracl
	}

	private CliContext ctx;

	@Option(names = "--logtype", required = false, split = ",", description = "comma-separated logtypes to query : calendar, mailbox_records, login, containeracl,...")
	public List<LogType> logTypes;

	@Option(names = "--key", required = false, description = "the auditlog entry key to query")
	public String messageId;

	@Option(names = "--with", required = false, description = "user mail address that must be present in auditlog entries")
	public String with;

	@Option(names = "--description", required = false, description = "event description that must be present in auditlog entries")
	public String description;

	@Option(names = "--domain", required = true, description = "domain uid or domain aliases")
	public String domainUid;

	@ArgGroup(exclusive = true)
	ItemOption itemOption;

	@ArgGroup(exclusive = true)
	ContainerOption containerOption;

	@Option(names = "--output", description = "output format to use (default ${DEFAULT-VALUE}): ${COMPLETION-CANDIDATES}")
	public OutputFormat outputFormat = OutputFormat.json;

	@Option(names = "--size", description = "number of entries to retrieve (default ${DEFAULT-VALUE})")
	public int size = 100;

	@Option(names = "--after", required = false, description = "get entries older than given date, format: 'yyyy-MM-dd'")
	public Date after;

	@Option(names = "--before", required = false, description = "get entries younger than given date, format: 'yyyy-MM-dd'")
	public Date before;

	@Option(names = "--owner", description = "the container owner address mail")
	public String owner;

	private static class ItemOption {
		@Option(names = "--item-id", required = true, description = "item id")
		long itemId;

		@Option(names = "--item-uid", required = true, description = "item uid")
		public String itemUid;
	}

	private static class ContainerOption {
		@Option(names = "--container-uid", required = true, description = "container uid")
		public String containerUid;

		@Option(names = "--container-id", required = true, description = "container id")
		public long containerId;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("auditlog");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return GetGLobalAuditLogCommand.class;
		}

	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	@Override
	public void run() {
		ILogRequestService service = ctx.adminApi().instance(ILogRequestService.class);
		AuditLogQuery auditLogQuery = new AuditLogQuery();

		IDomains domainsApi = ctx.adminApi().instance(IDomains.class);
		ItemValue<Domain> domain = domainsApi.findByNameOrAliases(domainUid);
		if (domain == null) {
			ctx.error("domain '" + domainUid + "' cannot be found");
			throw new CliException("domain '" + domainUid + "' cannot be found");
		}
		auditLogQuery.domainUid = domain.uid;

		auditLogQuery.size = size;
		if (logTypes != null && !logTypes.isEmpty()) {
			auditLogQuery.logtypes = logTypes.stream().map(l -> l.name()).toArray(String[]::new);
		}

		if (!StringUtil.isNullOrEmpty(with)) {
			auditLogQuery.with = with;
		}

		if (!StringUtil.isNullOrEmpty(description)) {
			auditLogQuery.description = description;
		}

		if (itemOption != null && !StringUtil.isNullOrEmpty(itemOption.itemUid)) {
			auditLogQuery.itemUid = itemOption.itemUid;
		}

		if (itemOption != null && itemOption.itemId != 0) {
			auditLogQuery.itemId = itemOption.itemId;
		}

		if (containerOption != null && !StringUtil.isNullOrEmpty(containerOption.containerUid)) {
			auditLogQuery.containerUid = containerOption.containerUid;
		}

		if (containerOption != null && containerOption.containerId != 0) {
			auditLogQuery.containerId = containerOption.containerId;
		}

		if (after != null) {
			auditLogQuery.after = after.getTime();
		}

		if (before != null) {
			auditLogQuery.before = before.getTime();
		}

		if (owner != null && !owner.isBlank()) {
			auditLogQuery.containerOwner = owner;
		}

		List<AuditLogEntry> logMailQuery = service.queryAuditLog(auditLogQuery);

		if (outputFormat == OutputFormat.table) {
			String[] headers = { "Log Type", "Timestamp", "Action", "Security Context Owner", "Container Owner Mail",
					"Content Description", "Content Key", "Update Message" };
			String[][] data = new String[logMailQuery.size()][headers.length];
			int index = 0;
			for (AuditLogEntry log : logMailQuery) {
				data[index][0] = log.logtype;
				data[index][1] = log.timestamp.toString();
				data[index][2] = log.action;
				data[index][3] = (log.securityContext != null) ? log.securityContext.email() : "";
				data[index][4] = (log.container != null) ? log.container.ownerElement().email() : "";
				data[index][5] = (log.content != null) ? log.content.description() : "";
				data[index][6] = (log.content != null) ? log.content.key() : "";
				data[index][7] = log.updatemessage;
				index++;
			}
			ctx.info(AsciiTable.getTable(headers, data));
		} else {
			logMailQuery.forEach(e -> ctx.info(JsonUtils.asString(e)));
		}
	}
}
