/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */

package net.bluemind.cli.auditlog;

import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.core.auditlogs.IAuditLogMgmt;
import net.bluemind.core.auditlogs.exception.AuditLogILMPolicyException;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "set-retention", description = "set retention duration (days) for datastream")
public class SetRetentionDurationAuditLogCommand implements Runnable, ICmdLet {
	private static final int MAX_VAL = 30;

	private static final Logger logger = LoggerFactory.getLogger(SetRetentionDurationAuditLogCommand.class);

	@Option(names = {
			"--value" }, required = true, description = "retention time duration (in days) - cannot be greater than 30 days")
	public int value = 3;

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("auditlog");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return SetRetentionDurationAuditLogCommand.class;
		}

	}

	private CliContext ctx;

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	@Override
	public void run() {
		IAuditLogMgmt service = ctx.adminApi().instance(IAuditLogMgmt.class);
		if (value > MAX_VAL) {
			logger.info("Wanted retention {} days is greater than {}", value, MAX_VAL);
			ctx.error("Failed: wanted retention " + value + " days is greater than " + MAX_VAL);
			return;
		}
		try {
			service.updateILMPolicyRetentionDuration(value);
			logger.info("Retention policy for auditlog successfully changed to {} days", value);
			ctx.info("Retention policy for auditlog successfully changed to " + value + " days");
		} catch (AuditLogILMPolicyException e) {
			logger.error("Cannot set retention policy for auditlog: {}", e.getMessage());
			ctx.error("Failed to set new rentention policy");
		}

	}

}
