/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.cli.auditlog;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.core.auditlogs.AuditLogEntry;
import net.bluemind.core.auditlogs.AuditLogQuery;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.domain.api.Domain;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;
import picocli.CommandLine.Command;

@Command(name = "mailfilter", description = "get mail filter rule data")
public class GetMailFilterRuleAuditLogCommand extends AbstractGetAuditLogCommand {

	private static final String[] HEADERS = { "Timestamp", "Action", "Action responsible", "origin", "value" };
	private static final String LOG_TYPE = "mail_filter_rule";

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("auditlog");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return GetMailFilterRuleAuditLogCommand.class;
		}

	}

	@Override
	protected String[][] generateOutputTable(List<AuditLogEntry> logMailQueryResults) {
		String[][] data = new String[logMailQueryResults.size()][HEADERS.length];
		int index = 0;
		for (AuditLogEntry log : logMailQueryResults) {
			data[index][0] = log.timestamp.toString();
			data[index][1] = log.action;
			data[index][2] = (log.securityContext != null) ? log.securityContext.email() : "";
			data[index][3] = (log.securityContext != null) ? log.securityContext.origin() : "";
			data[index][4] = (log.content != null) ? log.content.newValue() : "";
			index++;
		}
		return data;
	}

	@Override
	protected Map<String, String> generateOutputJSON(AuditLogEntry result) {
		Map<String, String> map = new HashMap<>();
		map.put("timestamp", result.timestamp.toString());
		map.put("action", result.action);
		map.put("securitycontext_owner", (result.securityContext != null) ? result.securityContext.email() : "");
		map.put("securitycontext_origin", (result.securityContext != null) ? result.securityContext.origin() : "");
		map.put("value", (result.content != null) ? result.content.newValue() : "");
		return map;
	}

	@Override
	protected AuditLogQuery buildQuery(ItemValue<Domain> domain) {
		AuditLogQuery auditLogQuery = defaultQuery();

		if (owner != null && !owner.isBlank()) {
			IUser userService = ctx.adminApi().instance(IUser.class, domain.uid);
			ItemValue<User> user = userService.byEmail(owner);
			if (user == null) {
				ctx.error("user with email '" + owner + "'cannot be found");
				throw new CliException("user with email '" + owner + "'cannot be found");
			}
			auditLogQuery.itemUid = user.uid;
			auditLogQuery.containerOwner = null;
		}
		return auditLogQuery;
	}

	@Override
	protected String[] headers() {
		return HEADERS;
	}

	@Override
	protected String[] logTypes() {
		return new String[] { LOG_TYPE };
	}
}
