/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.central.reverse.proxy.launcher;

import java.util.concurrent.CompletableFuture;

import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import net.bluemind.central.reverse.proxy.ReverseProxyServer;
import net.bluemind.central.reverse.proxy.common.config.CrpConfig;
import net.bluemind.central.reverse.proxy.stream.DirEntriesStreamVerticleFactory;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.system.application.registration.model.ApplicationInfo;
import net.bluemind.systemd.notify.SystemD;
import sun.misc.Signal;

public class CRPLauncher implements IApplication {

	private static final Logger logger = LoggerFactory.getLogger(CRPLauncher.class);
	private final Config config = CrpConfig.get("CRP", getClass().getClassLoader());

	@Override
	public Object start(IApplicationContext context) throws Exception {
		Thread.setDefaultUncaughtExceptionHandler(
				(thread, exception) -> logger.error("Unhandled exception in {}", thread, exception));

		ReverseProxyServer crp = new ReverseProxyServer();
		CompletableFuture<Void> prom = crp.run();

		var shutdownPublisher = VertxPlatform.eventBus().publisher("shutdown.requested");

		Signal.handle(new Signal("TERM"), s -> shutdownPublisher.write(null));
		Signal.handle(new Signal("INT"), s -> shutdownPublisher.write(null));

		// we notify systemd early because we can wait for a very long time for broker &
		// topics to become available
		if (SystemD.isAvailable()) {
			SystemD.get().notifyReady();
		}
		return prom.thenRun(() -> {
			registerApplication();
			logger.info("CRP started");
		}).exceptionally(t -> {
			logger.error(t.getMessage(), t);
			return null;
		}).thenApply(v -> IApplication.EXIT_OK).join();
	}

	private void registerApplication() {
		ApplicationInfo.register(DirEntriesStreamVerticleFactory.config.getString(CrpConfig.Stream.FOREST_ID));
	}

	@Override
	public void stop() {
		logger.info("Application stopped.");
	}

}
