/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.utils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.Semaphore;
import java.util.function.Consumer;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Utility methods to extract data from a DOM.
 * 
 * 
 */
public final class DOMUtils {
	private static final Logger logger = LoggerFactory.getLogger(DOMUtils.class);

	private static TransformerFactory fac;
	private static DocumentBuilderFactory dbf;
	private static DocumentBuilder mainbuilder;
	private static Semaphore builderLock;

	static {
		fac = TransformerFactory.newInstance();
		dbf = DocumentBuilderFactory.newInstance();
		dbf.setNamespaceAware(true);
		dbf.setValidating(false);
		try {
			mainbuilder = dbf.newDocumentBuilder();
			builderLock = new Semaphore(1);
		} catch (ParserConfigurationException e) {
		}
	}

	private static final LockingDocumentBuilder lock() {
		boolean acc = builderLock.tryAcquire();
		if (acc) {
			return () -> mainbuilder;
		} else {
			return new LockingDocumentBuilder() {

				@Override
				public DocumentBuilder builder() {
					try {
						synchronized (dbf) {
							return dbf.newDocumentBuilder();
						}
					} catch (ParserConfigurationException e) {
						return null;
					}
				}

				@Override
				public void unlock() {
				}
			};
		}
	}

	private static final void unlock() {
		builderLock.release();
	}

	public static String getElementText(Element root, String elementName) {
		NodeList list = root.getElementsByTagName(elementName);
		if (list.getLength() == 0) {
			logger.error("No element named '{}' under '{}'", elementName, root.getNodeName());
			return null;
		}
		return getElementText((Element) list.item(0));
	}

	public static String getElementText(Element node) {
		Text txtElem = (Text) node.getFirstChild();
		if (txtElem == null) {
			return null;
		}
		return txtElem.getData();
	}

	public static String[] getTexts(Element root, String elementName) {
		NodeList list = root.getElementsByTagName(elementName);
		String[] ret = new String[list.getLength()];
		for (int i = 0; i < list.getLength(); i++) {
			Text txt = (Text) list.item(i).getFirstChild();
			if (txt != null) {
				ret[i] = txt.getData();
			} else {
				ret[i] = ""; //$NON-NLS-1$
			}
		}
		return ret;
	}

	public static int countElementsForTag(Element root, String tagName) {
		NodeList list = root.getElementsByTagName(tagName);
		return (list == null) ? 0 : list.getLength();
	}

	/**
	 * Renvoie sous la forme d'un tableau la valeur des attributs donnés pour toutes
	 * les occurences d'un élément donnée dans le dom
	 * 
	 * <code>
	 *  <toto>
	 *   <titi id="a" val="ba"/>
	 *   <titi id="b" val="bb"/>
	 *  </toto>
	 * </code>
	 * 
	 * et getAttributes(&lt;toto&gt;, "titi", { "id", "val" }) renvoie { { "a", "ba"
	 * } { "b", "bb" } }
	 * 
	 * @param root
	 * @param elementName
	 * @param wantedAttributes
	 * @return
	 */
	public static String[][] getAttributes(Element root, String elementName, String[] wantedAttributes) {
		NodeList list = root.getElementsByTagName(elementName);
		String[][] ret = new String[list.getLength()][wantedAttributes.length];
		for (int i = 0; i < list.getLength(); i++) {
			Element elem = (Element) list.item(i);
			for (int j = 0; j < wantedAttributes.length; j++) {
				ret[i][j] = elem.getAttribute(wantedAttributes[j]);
			}
		}
		return ret;
	}

	/**
	 * Renvoie la valeur de l'attribut donné, d'un élément donné qui doit être
	 * unique sous l'élément racine
	 * 
	 * @param root
	 * @param elementName
	 * @param attribute
	 * @return
	 */
	public static String getElementAttribute(Element root, String elementName, String attribute) {
		NodeList list = root.getElementsByTagName(elementName);
		if (list.getLength() == 0) {
			return null;
		}
		return ((Element) list.item(0)).getAttribute(attribute);
	}

	/**
	 * Renvoie une élément qui doit être unique dans le document.
	 * 
	 * @param root
	 * @param elementName
	 * @return
	 */
	public static Element getUniqueElement(Element root, String elementName) {
		NodeList list = root.getElementsByTagName(elementName);
		return (Element) list.item(0);
	}

	public static void forEachElement(Element root, String elementName, Consumer<Element> cons) {
		NodeList list = root.getElementsByTagName(elementName);
		int len = list.getLength();
		for (int i = 0; i < len; i++) {
			cons.accept((Element) list.item(i));
		}
	}

	public static Element findElementWithUniqueAttribute(Element root, String elementName, String attribute,
			String attributeValue) {
		NodeList list = root.getElementsByTagName(elementName);
		for (int i = 0; i < list.getLength(); i++) {
			Element tmp = (Element) list.item(i);
			if (tmp.getAttribute(attribute).equals(attributeValue)) {
				return tmp;
			}
		}
		return null;
	}

	/**
	 * This method ensures that the output String has only valid XML unicode
	 * characters as specified by the XML 1.0 standard. For reference, please see
	 * <a href="http://www.w3.org/TR/2000/REC-xml-20001006#NT-Char">the
	 * standard</a>. This method will return an empty String if the input is null or
	 * empty.
	 * 
	 * @param in The String whose non-valid characters we want to remove.
	 * @return The in String, stripped of non-valid characters.
	 */
	public static final String stripNonValidXMLCharacters(String in) {
		char[] current = in.toCharArray();
		StringBuilder out = new StringBuilder(current.length);

		for (int i = 0; i < current.length; i++) {
			char c = current[i];
			if (validXmlChar((int) c)) {
				out.append(c);
			}
		}
		return out.toString();
	}

	// Char ::= #x9 | #xA | #xD | [#x20-#xD7FF] | [#xE000-#xFFFD] |
	// [#x10000-#x10FFFF]
	private static final boolean validXmlChar(int c) {
		return c == 0x9 || c == 0xA || c == 0xD || (c >= 0x20 && c <= 0xD7FF) || (c >= 0xE000 && c <= 0xFFFD)
				|| (c >= 0x10000 && c <= 0x10FFFF);
	}

	public static Element createElementAndText(Element parent, String elementName, String text) {
		if (text == null) {
			throw new NullPointerException("null text");
		}
		Element el = parent.getOwnerDocument().createElement(elementName);
		setElementText(parent, text, el);
		return el;
	}

	public static Element createElementAndText(Element parent, String namespace, String elementName, String text) {
		if (text == null) {
			throw new NullPointerException("null text");
		}
		Element el = parent.getOwnerDocument().createElementNS(namespace, elementName);
		setElementText(parent, text, el);
		return el;
	}

	private static void setElementText(Element parent, String text, Element el) {
		parent.appendChild(el);
		Text txt = el.getOwnerDocument().createTextNode(stripNonValidXMLCharacters(text));
		el.appendChild(txt);
	}

	public static Element createElement(Element parent, String elementName) {
		Element el = parent.getOwnerDocument().createElement(elementName);
		parent.appendChild(el);
		return el;
	}

	private static void serialise(Document doc, OutputStream out, boolean pretty) throws TransformerException {
		Transformer tf = fac.newTransformer();
		if (pretty) {
			tf.setOutputProperty(OutputKeys.INDENT, "yes");
			tf.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
		} else {
			tf.setOutputProperty(OutputKeys.INDENT, "no");
		}
		Source input = new DOMSource(doc.getDocumentElement());
		Result output = new StreamResult(out);
		tf.transform(input, output);
	}

	public static void serialise(Document doc, OutputStream out) throws TransformerException {
		serialise(doc, out, false);
	}

	public static String logDom(Document doc) throws TransformerException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		serialise(doc, out, true);
		String ret = out.toString();
		return ret;
	}

	public static Document parse(InputStream is)
			throws SAXException, IOException, ParserConfigurationException, FactoryConfigurationError {
		LockingDocumentBuilder builder = lock();
		Document ret = null;
		try {
			ret = builder.builder().parse(is);
		} finally {
			builder.unlock();
		}
		return ret;
	}

	public static Document parse(File f)
			throws SAXException, IOException, ParserConfigurationException, FactoryConfigurationError {
		LockingDocumentBuilder builder = lock();
		Document ret = null;
		try {
			ret = builder.builder().parse(f);
		} finally {
			builder.unlock();
		}
		return ret;
	}

	public static Document createDoc(String namespace, String rootElement)
			throws ParserConfigurationException, FactoryConfigurationError {
		LockingDocumentBuilder builder = lock();
		DocumentBuilder domBuilder = builder.builder();
		DOMImplementation di = domBuilder.getDOMImplementation();
		Document ret = null;
		try {
			ret = di.createDocument(namespace, rootElement, null);
		} finally {
			builder.unlock();
		}
		return ret;
	}

	public static void saxParse(InputStream is, DefaultHandler handler)
			throws SAXException, IOException, ParserConfigurationException {
		SAXParserFactory parserFactory = SAXParserFactory.newInstance();
		SAXParser parser = parserFactory.newSAXParser();
		XMLReader reader = parser.getXMLReader();
		reader.setContentHandler(handler);
		reader.setErrorHandler(handler);
		reader.parse(new InputSource(is));
	}

	private interface LockingDocumentBuilder {
		public default void unlock() {
			DOMUtils.unlock();
		}

		public DocumentBuilder builder();
	}
}
