/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.user.api;

import java.util.List;

import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.api.BMApi;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IRestoreDirEntryWithMailboxSupport;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.task.api.TaskRef;

@BMApi(version = "3")
@Path("/shared_mailboxes/{domainUid}")
public interface ISharedMailbox extends IRestoreDirEntryWithMailboxSupport<SharedMailbox> {

	/**
	 * Fetches a {@link SharedMailbox} by its unique id
	 *
	 * @param uid the shared mailbox's unique id
	 * @return the shared mailbox item value, or null if the shared mailbox does not exist
	 * @throws ServerFault standard error object (unchecked exception)
	 */
	@GET
	@Path("{uid}/complete")
	public ItemValue<SharedMailbox> getComplete(@PathParam(value = "uid") String uid) throws ServerFault;

	/**
	 * Fetches a {@link SharedMailbox} by its unique id, without vcard or mailbox
	 *
	 * @param uid
	 * @return
	 * @throws ServerFault
	 */
	@GET
	@Path("{uid}/light")
	ItemValue<SharedMailbox> getLight(@PathParam(value = "uid") String uid) throws ServerFault;

	/**
	 * Fetches a {@link SharedMailbox} by its email
	 *
	 * @param email the shared mailbox's email
	 * @return the shared mailbox item value, or null if the shared mailbox does not exist
	 * @throws ServerFault standard error object (unchecked exception)
	 */
	@GET
	@Path("byEmail/{email}")
	public ItemValue<SharedMailbox> byEmail(@PathParam(value = "email") String email) throws ServerFault;

	/**
	 * Fetches a {@link SharedMailbox} by its name
	 *
	 * @param login the shared mailbox's login/name
	 * @return the shared mailbox item value, or null if the shared mailbox does not exist
	 * @throws ServerFault standard error object (unchecked exception)
	 */
	@GET
	@Path("byName/{login}")
	public ItemValue<SharedMailbox> byName(@PathParam(value = "login") String login) throws ServerFault;

	/**
	 * Deletes a {@link SharedMailbox}. Also deletes all shared mailbox related objects and
	 * the shared mailbox's mailbox
	 *
	 * @param uid the shared mailbox's unique id
	 * @throws ServerFault standard error object (unchecked exception)
	 */
	@DELETE
	@Path("{uid}")
	public TaskRef delete(@PathParam(value = "uid") String uid) throws ServerFault;

	/**
	 * Retrieve all existing {@link SharedMailbox} uids
	 *
	 * @return a list of all existing {@link SharedMailbox} uids
	 * @throws ServerFault standard error object (unchecked exception)
	 */
	@GET
	@Path("_alluids")
	public List<String> allUids() throws ServerFault;

	/**
	 * Retrieves a {@link SharedMailbox}'s contact informations
	 * ({@link net.bluemind.addressbook.api.VCard})
	 *
	 * @param uid shared mailbox's unique id
	 * @return the shared mailbox's contact informations
	 * @throws ServerFault standard error object (unchecked exception)
	 */
	@GET
	@Path("{uid}/vcard")
	public VCard getVCard(@PathParam("uid") String uid) throws ServerFault;

}
