/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.hornetq.client.impl;

import java.net.SocketAddress;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.LongAdder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Iterables;

import io.lettuce.core.RedisChannelHandler;
import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisConnectionStateListener;
import io.lettuce.core.TrackingArgs;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.push.PushListener;
import io.lettuce.core.api.push.PushMessage;
import io.lettuce.core.codec.StringCodec;
import io.lettuce.core.pubsub.StatefulRedisPubSubConnection;
import io.netty.util.concurrent.DefaultThreadFactory;
import io.vertx.core.json.JsonObject;
import net.bluemind.keydb.common.ClientProvider;
import net.bluemind.keydb.common.JsonRedisCodec;

public class RedisConnection implements AutoCloseable {
	private static final Logger logger = LoggerFactory.getLogger(RedisConnection.class);
	private static final DefaultThreadFactory tf = new DefaultThreadFactory("keydb-consume-loop");
	private static final Iterator<ExecutorService> cyclic = Iterables.cycle(//
			Executors.newSingleThreadExecutor(tf), //
			Executors.newSingleThreadExecutor(tf), //
			Executors.newSingleThreadExecutor(tf), //
			Executors.newSingleThreadExecutor(tf)).iterator();

	private final RedisClient redisClient;
	private final StatefulRedisPubSubConnection<String, JsonObject> pubSubConnection;
	private final StatefulRedisConnection<String, String> commands;
	private final Map<String, Runnable> invalidationListeners = new ConcurrentHashMap<>();
	private final Map<String, Runnable> reconnectListeners = new ConcurrentHashMap<>();

	private final LongAdder invalidations = new LongAdder();

	public RedisConnection(String ip) {
		redisClient = ClientProvider.newClient(ip);
		this.pubSubConnection = redisClient.connectPubSub(new JsonRedisCodec());
		this.commands = redisClient.connect();

		initInvalidationListener();
		initConnectionListener();
	}

	private void initInvalidationListener() {
		PushListener pl = (PushMessage message) -> {
			String type = message.getType();
			if (type.equals("invalidate")) {
				invalidations.increment();
				@SuppressWarnings("unchecked")
				List<String> keys = (List<String>) message.getContent(StringCodec.ASCII::decodeKey).get(1);
				if (keys != null) {
					keys.forEach(k -> Optional.ofNullable(invalidationListeners.get(k))
							.ifPresent(r -> pushExecutor().submit(r)));
				}
			}
		};

		this.commands.addListener(pl);
		enableTracking();
	}

	private void initConnectionListener() {
		redisClient.addListener(new RedisConnectionStateListener() {
			@Override
			public void onRedisConnected(RedisChannelHandler<?, ?> connection, SocketAddress socketAddress) {
				if (connection == commands) {
					Thread.ofVirtual().start(() -> {
						enableTracking();
						reconnectListeners.forEach((k, v) -> pushExecutor().submit(v));
						logger.debug("[keydb] Reconnected, tracking set");
					});
				}
			}
		});
	}

	private void enableTracking() {
		commands.sync().clientTracking(TrackingArgs.Builder.enabled());
	}

	public ExecutorService pushExecutor() {
		return cyclic.next();
	}

	public long invalidationsReceived() {
		return invalidations.sum();
	}

	public StatefulRedisPubSubConnection<String, JsonObject> pubsub() {
		return this.pubSubConnection;
	}

	public void registerInvalidationListener(String k, Runnable r) {
		this.invalidationListeners.put(k, r);
	}

	public void registerReconnectListener(String k, Runnable r) {
		this.reconnectListeners.put(k, r);
	}

	public StatefulRedisConnection<String, String> keyvalue() {
		return commands;
	}

	public void close() {
		redisClient.close();
	}
}
