/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.utils;

import java.io.IOException;
import java.net.ConnectException;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.Socket;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpClient.Builder;
import java.net.http.HttpRequest;
import java.net.http.HttpRequest.BodyPublisher;
import java.net.http.HttpRequest.BodyPublishers;
import java.net.http.HttpResponse;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.Map;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509ExtendedTrustManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;

public class SyncHttpClient {
	private final Logger logger = LoggerFactory.getLogger(SyncHttpClient.class);

	private static final SyncHttpClient INSTANCE = new SyncHttpClient();

	private HttpClient httpClient = initHttpClient();

	public static final SyncHttpClient getInstance() {
		return INSTANCE;
	}

	private SyncHttpClient() {
	}

	private HttpClient initHttpClient() {
		Builder httpClientBuilder = HttpClient.newBuilder().version(HttpClient.Version.HTTP_2).proxy(proxy())
				.connectTimeout(Duration.ofSeconds(10));
		if (System.getProperty("net.bluemind.acceptAllCertificates") != null) {
			setSSLContext(httpClientBuilder);
		}

		return httpClientBuilder.build();
	}

	private ProxySelector proxy() {
		Map<String, String> sysConfMap = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(ISystemConfiguration.class).getValues().values;
		String proxyEnabled = sysConfMap.get(SysConfKeys.http_proxy_enabled.name());
		if (proxyEnabled == null || proxyEnabled.trim().isEmpty() || !proxyEnabled.equals("true")) {
			return ProxySelector.getDefault();
		} else {
			String host = sysConfMap.get(SysConfKeys.http_proxy_hostname.name());
			int port = Integer.parseInt(sysConfMap.get(SysConfKeys.http_proxy_port.name()));
			return ProxySelector.of(new InetSocketAddress(host, port));
		}

	}

	public String get(String url) {
		URI uri;
		try {
			uri = new URI(url);
		} catch (Exception e) {
			throw new ServerFault("Invalid URL");
		}

		BodyPublisher payload = BodyPublishers.noBody();
		HttpRequest httpRequest = HttpRequest.newBuilder() //
				.timeout(Duration.ofSeconds(10)) //
				.method("GET", payload) //
				.uri(uri) //
				.build();

		HttpResponse<String> httpResponse;
		try {
			httpResponse = httpClient.send(httpRequest, HttpResponse.BodyHandlers.ofString());

			int code = httpResponse.statusCode();
			if (code != 200) {
				throw new ServerFault("Response code error " + httpResponse.statusCode());
			}

			return httpResponse.body();
		} catch (IOException e) {
			if (logger.isDebugEnabled()) {
				logger.debug("Get " + url + " fail", e);
			}

			if (e.getCause() instanceof ConnectException) {
				throw new ServerFault("Unable to connect");
			}

			throw new ServerFault("Unable to get: " + e.getMessage());
		} catch (InterruptedException ie) {
			if (logger.isDebugEnabled()) {
				logger.debug("Get " + url + " fail", ie);
			}

			throw new ServerFault("Unable to get: interrupted", ie);
		}
	}

	private void setSSLContext(Builder httpClientBuilder) {
		var trustManager = new X509ExtendedTrustManager() {
			@Override
			public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
				// Do nothing
			}

			@Override
			public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
				// Do nothing
			}

			@Override
			public X509Certificate[] getAcceptedIssuers() {
				return new X509Certificate[0];
			}

			@Override
			public void checkClientTrusted(X509Certificate[] chain, String authType, Socket socket)
					throws CertificateException {
				// Do nothing
			}

			@Override
			public void checkServerTrusted(X509Certificate[] chain, String authType, Socket socket)
					throws CertificateException {
				// Do nothing
			}

			@Override
			public void checkClientTrusted(X509Certificate[] chain, String authType, SSLEngine engine)
					throws CertificateException {
				// Do nothing
			}

			@Override
			public void checkServerTrusted(X509Certificate[] chain, String authType, SSLEngine engine)
					throws CertificateException {
				// Do nothing
			}
		};

		SSLContext sslContext;
		try {
			sslContext = SSLContext.getInstance("TLS");
			sslContext.init(null, new TrustManager[] { trustManager }, new SecureRandom());
		} catch (NoSuchAlgorithmException | KeyManagementException e) {
			System.clearProperty("net.bluemind.acceptAllCertificates");
			return;
		}

		httpClientBuilder.sslContext(sslContext).build();
	}
}
