/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.utils;

import java.net.HttpURLConnection;
import java.net.Socket;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.List;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509ExtendedTrustManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.sysconf.helper.SysConfHelper;

/**
 * Helper class to disable SSL chains validation
 */
public class Trust {

	private static SSLSocketFactory defaultSocketFactory = HttpsURLConnection.getDefaultSSLSocketFactory();
	private static final Logger logger = LoggerFactory.getLogger(Trust.class);

	public Trust() {

	}

	public void prepareConnection(String module, HttpURLConnection con) {
		if (!(con instanceof HttpsURLConnection)) {
			return;
		}

		if (trustall(module)) {
			((HttpsURLConnection) con).setSSLSocketFactory(getSSLSocketFactory(true));
			((HttpsURLConnection) con).setHostnameVerifier(getHostNameVerifier(true));
		}
	}

	public SSLSocketFactory getSSLSocketFactory(String module) {
		return getSSLSocketFactory(trustall(module));
	}

	public SSLSocketFactory getSSLSocketFactory(boolean trustAll) {
		if (trustAll) {
			return createSSLContext().getSocketFactory();
		} else {
			return defaultSocketFactory;
		}
	}

	public SSLContext getSSLContext(String module) {
		return getSSLContext(trustall(module));
	}

	public SSLContext getSSLContext(boolean trustAll) {
		if (trustAll) {
			return createSSLContext();
		} else {
			try {
				return SSLContext.getDefault();
			} catch (NoSuchAlgorithmException e) {
				throw new Error("Failed to initialize the SSLContext", e);
			}
		}
	}

	private static final X509ExtendedTrustManager trustAll = new X509ExtendedTrustManager() {

		@Override
		public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
		}

		@Override
		public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
		}

		@Override
		public X509Certificate[] getAcceptedIssuers() {
			return new X509Certificate[0];
		}

		@Override
		public void checkClientTrusted(X509Certificate[] chain, String authType, Socket socket)
				throws CertificateException {
		}

		@Override
		public void checkServerTrusted(X509Certificate[] chain, String authType, Socket socket)
				throws CertificateException {
		}

		@Override
		public void checkClientTrusted(X509Certificate[] chain, String authType, SSLEngine engine)
				throws CertificateException {
		}

		@Override
		public void checkServerTrusted(X509Certificate[] chain, String authType, SSLEngine engine)
				throws CertificateException {
		}

	};

	private static final TrustManager[] trustManagers = new TrustManager[] { trustAll };

	/**
	 * Creates a trust-all ssl context
	 *
	 * @return an accept all context
	 */
	private SSLContext createSSLContext() {
		try {
			SecureRandom secureRandom = new SecureRandom();

			SSLContext sslContext = SSLContext.getInstance("TLS");
			sslContext.init(null, trustManagers, secureRandom);

			return sslContext;
		} catch (Exception e) {
			throw new Error("Failed to initialize the SSLContext", e);
		}
	}

	public final HostnameVerifier getHostNameVerifier(String module) {
		return getHostNameVerifier(trustall(module));
	}

	private final HostnameVerifier getHostNameVerifier(boolean trustAll) {
		if (trustAll) {
			return (hostname, session) -> true;
		} else {
			return HttpsURLConnection.getDefaultHostnameVerifier();
		}
	}

	private boolean trustall(String module) {
		List<String> modules = getTrustAllModules();
		if (modules.isEmpty()) {
			return false;
		}
		if (modules.size() == 1 && modules.get(0).equals("ALL")) {
			return true;
		}
		return modules.contains(module);
	}

	protected List<String> getTrustAllModules() {
		return SysConfHelper.fromSharedMap().stringList(SysConfKeys.tls_trust_allcertificates.name());
	}

}
