/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.hornetq.client.impl;

import java.net.NetworkInterface;
import java.net.SocketException;
import java.time.Duration;
import java.util.Collections;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.hornetq.client.MQ.IMQConnectHandler;
import net.bluemind.hornetq.client.MQKeyDB;
import net.bluemind.keydb.common.KeydbBootstrapNetAddress;

public class KeyDBClient {
	protected final Logger logger = LoggerFactory.getLogger(getClass());
	public final CompletableFuture<MQKeyDB> keyDBStart = new CompletableFuture<>();

	public KeyDBClient() {
		String keyDbIp = KeydbBootstrapNetAddress.getKeydbIP();
		final String connectionTarget;

		Config coreConfig = CoreConfig.get();

		if (isLocalHost(keyDbIp)) {
			connectionTarget = "redis-socket://" + coreConfig.getString(CoreConfig.Redis.SOCKET_PATH);
			logger.info("KeyDB host '{}' detected as local. Attempting connection via Unix socket: {}", keyDbIp,
					connectionTarget);
		} else {
			connectionTarget = keyDbIp;
			logger.info("Connecting to KeyDB at {}....", connectionTarget);
		}

		logger.info("Connecting to {}....", connectionTarget);
		ForkJoinPool forComp = ForkJoinPool.commonPool();
		Thread.ofPlatform().name("bm-keydb-connect").start(() -> {
			MQKeyDB mq = null;
			do {
				try {
					mq = new MQKeyDB(new RedisConnection(connectionTarget));
					final MQKeyDB obtained = mq;
					forComp.submit(() -> keyDBStart.complete(obtained));
					logger.info("Keydb connection setup completed ({})", connectionTarget);
				} catch (Exception e) {
					logger.error(e.getMessage(), e);
					try {
						Thread.sleep(Duration.ofSeconds(5));
					} catch (InterruptedException ie) {
						Thread.currentThread().interrupt();
						forComp.submit(() -> keyDBStart.completeExceptionally(ie));
						break;
					}
				}
			} while (mq == null);
		});
	}

	public CompletableFuture<Void> init() {
		CompletableFuture<Void> initFuture = new CompletableFuture<>();
		keyDBStart.whenComplete((keydb, ex) -> {
			if (ex == null) {
				initFuture.complete(null);
			} else {
				initFuture.completeExceptionally(ex);
			}
		});
		return initFuture;
	}

	public final synchronized void init(final IMQConnectHandler handler) {
		keyDBStart.whenComplete((keydb, ex) -> {
			if (ex == null) {
				handler.connected();
			}
		});
	}

	/**
	 * Checks if the given host address corresponds to a local network interface.
	 * 
	 * @param host The hostname or IP address to check.
	 * @return {@code true} if the host is a local address, {@code false} otherwise.
	 */
	private boolean isLocalHost(String host) {
		if (host == null || host.trim().isEmpty()) {
			return false;
		}
		// Handle localhost names explicitly for robustness
		if ("localhost".equalsIgnoreCase(host) || "127.0.0.1".equals(host) || "::1".equals(host)) {
			return true;
		}

		try {
			// Get all network interfaces of this machine
			return Collections.list(NetworkInterface.getNetworkInterfaces()).stream()
					.flatMap(iface -> Collections.list(iface.getInetAddresses()).stream())
					.anyMatch(inetAddress -> inetAddress.getHostAddress().equals(host));
		} catch (SocketException e) {
			logger.error("Could not check network interfaces to determine if host {} is local", host, e);
			return false; // Fail safe to false
		}
	}
}
