/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.hornetq.client;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.Timer;

import io.lettuce.core.pubsub.RedisPubSubAdapter;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.hornetq.client.MQ.SharedMap;
import net.bluemind.hornetq.client.impl.RedisConnection;
import net.bluemind.metrics.registry.IdFactory;

public class MQKeyDB implements AutoCloseable {

	public static record ConsumerRegistration(OutOfProcessMessageHandler handler, Predicate<JsonObject> filter,
			ExecutorService deliveryLoop) {

		public void deliver(long startNanos, Timer timer, JsonObject msg) {
			deliveryLoop.submit(() -> {
				long inPoolLatency = System.nanoTime() - startNanos;
				timer.record(inPoolLatency, TimeUnit.NANOSECONDS);
				if (filter.test(msg)) {
					handler.handle(new OOPMessage(msg));
				}
			});
		}
	}

	private final Map<String, List<ConsumerRegistration>> consumers = new ConcurrentHashMap<>();
	private final Map<String, Producer> producersTopics = new ConcurrentHashMap<>();
	private final RedisConnection redisConnection;
	private final Registry reg;
	private final IdFactory idFactory;

	public MQKeyDB(RedisConnection redis) {
		this.redisConnection = redis;
		reg = net.bluemind.metrics.registry.MetricsRegistry.get();
		idFactory = new IdFactory("keydb", reg, MQKeyDB.class);

		redisConnection.pubsub().addListener(new RedisPubSubAdapter<String, JsonObject>() {
			@Override
			public void message(String topic, JsonObject message) {
				Timer timer = reg.timer(idFactory.name("pubsub.latency", "topic", topic));
				long start = System.nanoTime();
				Optional.ofNullable(consumers.get(topic)).ifPresent(handlers -> handlers//
						.stream()//
						.forEach(consumerReg -> consumerReg.deliver(start, timer, message)));
			}
		});
	}

	public Consumer registerConsumer(String topic, OutOfProcessMessageHandler handler) {
		return registerConsumer(topic, null, handler);
	}

	public Consumer registerConsumer(String topic, Predicate<JsonObject> filter, OutOfProcessMessageHandler handler) {
		return getConsumer(topic, filter, handler);
	}

	public Consumer getConsumer(String topic, Predicate<JsonObject> filter, OutOfProcessMessageHandler handler) {
		List<ConsumerRegistration> listMsg = consumers.computeIfAbsent(topic, key -> {
			List<ConsumerRegistration> newListMsg = new CopyOnWriteArrayList<>();
			redisConnection.pubsub().sync().subscribe(topic);
			return newListMsg;
		});

		Predicate<JsonObject> outOfProcessMessageFilter = filter == null ? v -> true : filter;
		ConsumerRegistration message = new ConsumerRegistration(handler, outOfProcessMessageFilter,
				redisConnection.pushExecutor());
		listMsg.add(message);

		return new Consumer(() -> listMsg.remove(message));
	}

	public Producer registerProducer(String topic) {
		return getProducer(topic);
	}

	public Producer getProducer(String topic) {
		return producersTopics.computeIfAbsent(topic, this::createProducer);
	}

	public Producer createProducer(String topic) {
		return new Producer(redisConnection.pubsub().async(), topic);
	}

	public OOPMessage newMessage() {
		return new OOPMessage(new JsonObject());
	}

	public static interface Codec<V> {
		public static final Codec<String> STR = new Codec<String>() {
			@Override
			public String toString(String value) {
				return value;
			}

			@Override
			public String fromString(String s) {
				return s;
			}
		};

		public static <C> Codec<C> forClass(Class<C> klass) {
			return new Codec<C>() {
				@Override
				public C fromString(String s) {
					return JsonUtils.read(s, klass);
				}

				@Override
				public String toString(C value) {
					return JsonUtils.asString(value);
				}
			};
		}

		V fromString(String s);

		String toString(V value);
	}

	public <K, V> SharedMap<K, V> sharedMap(String name, Codec<K> forKey, Codec<V> forValue) {
		return new KeyDBSharedMap<>(name, forKey, forValue, redisConnection);
	}

	public <C> SharedMap<String, C> sharedMap(String name, Class<C> forClass) {
		return new KeyDBSharedMap<>(name, Codec.STR, Codec.forClass(forClass), redisConnection);
	}

	public SharedMap<String, String> sharedMap(String name) {
		return new KeyDBSharedMap<>(name, Codec.STR, Codec.STR, redisConnection);
	}

	public Map<String, List<ConsumerRegistration>> getConsumers() {
		return consumers;
	}

	public RedisConnection getRedisConnection() {
		return redisConnection;
	}

	@Override
	public void close() {
		redisConnection.close();
	}

	public Registry registry() {
		return reg;
	}

	public Map<String, Producer> getProducersTopics() {
		return producersTopics;
	}
}
