/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.utils;

import java.text.DecimalFormat;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

public class ProgressPrinter {
	private final long total;
	private long start;
	private AtomicLong elements = new AtomicLong(0);
	private final long printEvery;
	private final long printEverySeconds;
	private final long printEverySecondsWhenFinished;
	private long printEveryPercent = -1;
	private Instant lastPrint = Instant.now();
	private static final DecimalFormat DEC_FORMAT = new DecimalFormat("#.##");

	private static final long KB = 1L * 1000;
	private static final long MB = KB * 1000;
	private static final long GB = MB * 1000;
	private static final long TB = GB * 1000;
	private static final long PB = TB * 1000;
	private static final long EB = PB * 1000;

	private static final long WINDOW_SIZE_MILLIS = Duration.ofMinutes(1).toMillis();
	private final CircularBuffer buffer;

	public ProgressPrinter(long total) {
		this(total, 10_000L, 2);
	}

	/**
	 * Generate a "progress" log every "printEvery" item, or every printEverySecond
	 * seconds
	 *
	 * @param total             long: total number of expected elements
	 * @param printEvery:       print the log every "n" elements
	 * @param printEverySeconds
	 */
	public ProgressPrinter(long total, long printEvery, long printEverySeconds, long printEverySecondsWhenFinished) {
		this(total, printEvery, printEverySeconds, -1, printEverySecondsWhenFinished);
	}

	public ProgressPrinter(long total, long printEvery, long printEverySeconds) {
		this(total, printEvery, printEverySeconds, -1, 120);
	}

	/**
	 * Generate a "progress" log every "printEvery" item, or every printEverySecond,
	 * or every printEveryPercent
	 *
	 * @param total             long: total number of expected elements
	 * @param printEvery:       print the log every "n" elements
	 * @param printEverySeconds
	 * @param printEveryPercent
	 */
	private ProgressPrinter(long total, long printEvery, long printEverySeconds, long printEveryPercent,
			long printEverySecondsWhenFinished) {
		this.total = total;
		this.printEvery = printEvery;
		this.printEverySeconds = printEverySeconds;
		this.printEveryPercent = printEveryPercent;
		this.printEverySecondsWhenFinished = printEverySecondsWhenFinished;
		this.start = System.nanoTime();
		this.buffer = new CircularBuffer(WINDOW_SIZE_MILLIS);
	}

	public static ProgressPrinter createWithPercent(long total, long printEveryPercent) {
		ProgressPrinter progressPrinter = new ProgressPrinter(total);
		progressPrinter.printEveryPercent = printEveryPercent;
		return progressPrinter;
	}

	public void reset() {
		start = System.nanoTime();
	}

	public void add() {
		long count = elements.incrementAndGet();
		buffer.add(count, System.currentTimeMillis());
	}

	public void add(long count) {
		long newCount = elements.addAndGet(count);
		buffer.add(newCount, System.currentTimeMillis());
	}

	private String plural(long v, String s) {
		return v + " " + s + (v > 1 ? "s" : "");
	}

	private String formatDuration(Duration duration) {
		List<String> parts = new ArrayList<>();
		long days = duration.toDaysPart();
		if (days > 0) {
			parts.add(plural(days, "day"));
		}
		int hours = duration.toHoursPart();
		if (hours > 0 || !parts.isEmpty()) {
			parts.add(plural(hours, "hour"));
		}
		int minutes = duration.toMinutesPart();
		if (minutes > 0 || !parts.isEmpty()) {
			parts.add(plural(minutes, "minute"));
		}
		int seconds = duration.toSecondsPart();
		if (seconds < 0) {
			parts.add("immediately");
		} else {
			parts.add(plural(seconds, "second"));
		}
		return String.join(", ", parts);
	}

	public static String toHumanReadableSIPrefixes(long size) {
		if (size < 0)
			throw new IllegalArgumentException("Invalid file size: " + size);
		if (size >= EB)
			return formatSize(size, EB, "E");
		if (size >= PB)
			return formatSize(size, PB, "P");
		if (size >= TB)
			return formatSize(size, TB, "T");
		if (size >= GB)
			return formatSize(size, GB, "G");
		if (size >= MB)
			return formatSize(size, MB, "M");
		if (size >= KB)
			return formatSize(size, KB, "K");
		return formatSize(size, 1L, "");
	}

	private static String formatSize(long size, long divider, String unitName) {
		return DEC_FORMAT.format((double) size / divider) + " " + unitName;
	}

	public String toString() {
		try {
			lastPrint = Instant.now();
			var duration = Duration.ofNanos(System.nanoTime() - start);
			var currentElements = elements.get();
			var durationSeconds = duration.toSeconds();
			StringBuilder sb = new StringBuilder();
			sb.append(currentElements);
			if (total > 0) {
				if (currentElements > total) {
					sb.append(" / ").append(total);
				} else {
					var percentage = total > 0 ? ((currentElements / (double) total) * 100.0) : 100.0;
					sb.append(" / ").append(total);
					sb.append(" (").append(new DecimalFormat("00.0#").format(percentage)).append("%)");
				}
			}
			if (durationSeconds > 0) {
				double rate = buffer.getRate();
				sb.append(" in ").append(formatDuration(duration));
				sb.append(" rate: ").append(toHumanReadableSIPrefixes((long) rate)).append("/s");
				if (total > 0 && rate > 0) {
					sb.append(" eta: ")
							.append(formatDuration(Duration.ofSeconds((long) ((total - currentElements) / rate))));
				}
			}
			return sb.toString();
		} catch (Exception e) {
			return e.getMessage();
		}
	}

	private double percent(long currentElements) {
		return total > 0 ? ((currentElements / (double) total) * 100.0) : 100.0;
	}

	public boolean shouldPrint() {
		var currentElements = elements.get();
		// When the printer has finished / in continuous mode, be more gentle
		if (currentElements > total) {
			return printFinishedDuration();
		}
		if (currentElements == total || ((currentElements % printEvery) == 0)) {
			return true;
		}

		if (printEveryPercent > 0) {
			return ((percent(currentElements) % printEveryPercent) == 0) && printDuration();
		}

		return printDuration();
	}

	private boolean printDuration() {
		return Duration.between(lastPrint, Instant.now()).toSeconds() >= printEverySeconds;
	}

	private boolean printFinishedDuration() {
		return Duration.between(lastPrint, Instant.now()).toSeconds() >= printEverySecondsWhenFinished;
	}

	private static class CircularBuffer {
		private final long[] timestamps;
		private final long[] counts;
		private int head;
		private int tail;
		private final long windowSizeMillis;

		public CircularBuffer(long windowSizeMillis) {
			this.windowSizeMillis = windowSizeMillis;
			int size = 2000;
			this.timestamps = new long[size];
			this.counts = new long[size];
			this.head = 0;
			this.tail = -1;
		}

		public void add(long count, long timestamp) {
			if (tail == -1 || (tail + 1) % timestamps.length != head) {
				tail = (tail + 1) % timestamps.length;
			} else {
				head = (head + 1) % timestamps.length;
			}
			timestamps[tail] = timestamp;
			counts[tail] = count;

			// Supprimer les entrées obsolètes
			while (timestamp - timestamps[head] > windowSizeMillis) {
				head = (head + 1) % timestamps.length;
			}
		}

		public double getRate() {
			if (tail == -1 || head == tail) {
				return 0;
			}
			long oldestTimestamp = timestamps[head];
			long newestTimestamp = timestamps[tail];
			long oldestCount = counts[head];
			long newestCount = counts[tail];

			long timeDiff = newestTimestamp - oldestTimestamp;
			long countDiff = newestCount - oldestCount;

			if (timeDiff == 0) {
				return 0;
			}

			return countDiff / (timeDiff / 1000.0);
		}
	}

}