/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.dataprotect.api;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel.MapMode;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Set;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.server.api.Server;
import net.bluemind.utils.FileUtils;

public interface IBackupWorker {

	String getDataType();

	public default long getTransferredMegaBytes() {
		return 0;
	}

	boolean supportsTag(String tag);

	/**
	 * This is called before the backup starts. This phase is used to put the data
	 * in a backup-able state.
	 *
	 * For a database it means creating a dump, for a cyrus mail server it means
	 * stopping it, etc.
	 *
	 * @param ctx
	 * @param tag
	 * @param toBackup
	 * @throws ServerFault
	 */
	public default void prepareDataDirs(IDPContext ctx, PartGeneration partGen, ItemValue<Server> toBackup)
			throws ServerFault {
		// do nothing
	}

	/**
	 * This is called after <code>prepareDataDirs</code>. Rsync will be used to save
	 * those dirs. Symlinks are handled by the dataprotect code, so you don't need
	 * special code here to check if the dir war symlinked elsewhere.
	 *
	 * @return the dirs that need to be data protected
	 */
	Set<String> getDataDirs();

	/**
	 * This gets called once the data is saved with rsync. At this point you can
	 * remove the data you exported for backup, restart the services that you
	 * stopped, etc.
	 *
	 * @param ctx
	 * @param tag
	 * @param backedUp
	 * @throws ServerFault
	 */
	default void dataDirsSaved(IDPContext ctx, String tag, ItemValue<Server> backedUp) throws ServerFault {
		// do nothing
	}

	public default void restore(IDPContext ctx, PartGeneration part, Map<String, Object> params) throws ServerFault {
		// do nothing
	}

	public default void cleanup(IDPContext ctx, PartGeneration part, Map<String, Object> params) throws ServerFault {
		getDataDirs().stream().map(Paths::get).filter(Files::exists).forEach(p -> {
			try {
				FileUtils.deleteDataDir(p, true);
			} catch (IOException e) {
				ctx.error(e, "Unable to delete temporary directory {} content of backup", p.toString());
			}
		});
	}

	public static ByteBuf mmapOutput() {
		try {
			Path backingFile = Files.createTempFile("dp-worker", ".output");
			int capacity = Integer.MAX_VALUE;
			try (RandomAccessFile raf = new RandomAccessFile(backingFile.toFile(), "rw")) {
				raf.setLength(Integer.MAX_VALUE);
				MappedByteBuffer targetBuffer = raf.getChannel().map(MapMode.READ_WRITE, 0, capacity);
				ByteBuf wrapped = Unpooled.wrappedBuffer(targetBuffer);
				wrapped.writerIndex(0).readerIndex(0);
				Files.delete(backingFile);
				return wrapped;
			}
		} catch (IOException e) {
			throw new ServerFault(e);
		}
	}

}
