/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.backend.mail.replica.service.internal;

import java.util.List;
import java.util.stream.Collectors;

import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.MessageBody.Part;
import net.bluemind.backend.mail.api.MessageBody.RecipientKind;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.core.auditlogs.AuditLogUpdateStatus;
import net.bluemind.core.auditlogs.AuditLogUpdateStatus.MessageCriticity;
import net.bluemind.core.auditlogs.ContentElement;
import net.bluemind.core.auditlogs.ContentElement.ContentElementBuilder;
import net.bluemind.core.auditlogs.ILogMapperProvider;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ChangeLogEntry.Type;

public class DbMailboxRecordsAuditLogMapper implements ILogMapperProvider<MailboxRecord> {
	private final IDbMessageBodies bodiesApi;
	private static final String VOICEMAIL = "voicemail";
	private static final String INVITATION = "invitation";
	private static final String ATTACHMENTS = "attachments";

	private static final AuditLogUpdateStatus SEEN_UPD = new AuditLogUpdateStatus("SeenChanged",
			MessageCriticity.MINOR);

	public DbMailboxRecordsAuditLogMapper(BaseContainerDescriptor bcd, IDbMessageBodies bodiesApi) {
		this.bodiesApi = bodiesApi;
	}

	@Override
	public ContentElement createContentElement(MailboxRecord mboxrecord, Type type) {
		if (mboxrecord == null) {
			return null;
		}
		MessageBody fetchedBody = bodiesApi.get(mboxrecord.messageBody);
		if (fetchedBody != null) {
			return filterMessageBody(fetchedBody);
		}
		return null;
	}

	@Override
	public AuditLogUpdateStatus createUpdateMessage(MailboxRecord oldValue, MailboxRecord newValue) {
		List<MailboxItemFlag> oldFlags = oldValue.flags;
		List<MailboxItemFlag> newFlags = newValue.flags;
		List<MailboxItemFlag> removedFlags = oldFlags.stream().filter(element -> !newFlags.contains(element)).toList();
		List<MailboxItemFlag> addedFlags = newFlags.stream().filter(element -> !oldFlags.contains(element)).toList();
		if (isMinorDifference(removedFlags, addedFlags)) {
			return SEEN_UPD;
		}

		StringBuilder stringBuilder = new StringBuilder();
		if (!removedFlags.isEmpty()) {
			stringBuilder.append("Removed Flags:\n")
					.append(removedFlags.stream().map(MailboxItemFlag::toString).collect(Collectors.joining(",")))
					.append("\n");
		}
		if (!addedFlags.isEmpty()) {
			stringBuilder.append("Added Flags:\n")
					.append(addedFlags.stream().map(MailboxItemFlag::toString).collect(Collectors.joining(",")))
					.append("\n");
		}
		return new AuditLogUpdateStatus(stringBuilder.toString());
	}

	private ContentElement filterMessageBody(MessageBody body) {
		ContentElementBuilder builder = new ContentElement.ContentElementBuilder();
		builder.description(body.subject);
		builder.key(body.messageId);
		builder.author(body.recipients.stream().filter(r -> r.kind.equals(RecipientKind.Originator)).map(r -> r.address)
				.toList());
		List<String> recipientsList = body.recipients.stream().map(r -> r.address).toList();
		builder.with(recipientsList);
		if (!body.structure.attachments().isEmpty()) {
			builder.has.add(ATTACHMENTS);
		}
		List<String> headerNames = body.headers.stream().map(h -> h.name.toLowerCase()).toList();
		if (headerNames.contains("x-bm-event") && headerNames.contains("x-bm-rsvp")) {
			builder.has.add(INVITATION);
		}
		if (headerNames.contains("x-asterisk-callerid")) {
			builder.has.add(VOICEMAIL);
		}
		List<Part> attachments = body.structure.attachments();
		if (attachments != null && !attachments.isEmpty()) {
			attachments.forEach(a -> {
				if (a.fileName != null && !a.fileName.isBlank()) {
					builder.has.add(a.fileName);
				}
				if (a.hash != null && !a.hash.isBlank()) {
					builder.has.add(a.hash);
				}
			});
		}
		return builder.build();
	}

	private boolean isMinorDifference(List<MailboxItemFlag> removed, List<MailboxItemFlag> added) {
		return justSeenChanged(removed, added) || justSeenChanged(added, removed);
	}

	private boolean justSeenChanged(List<MailboxItemFlag> from, List<MailboxItemFlag> to) {
		return from.size() == 1 && to.isEmpty() && from.contains(MailboxItemFlag.System.Seen.value());
	}

}
