/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2017
 *
 * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License)
 * or the CeCILL as published by CeCILL.info (version 2 of the License).
 *
 * There are special exceptions to the terms and conditions of the
 * licenses as they are applied to this program. See LICENSE.txt in
 * the directory of this program distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.backend.mail.replica.service;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.function.Supplier;

import net.bluemind.backend.mail.api.IMailboxItems;
import net.bluemind.backend.mail.api.ImapAck;
import net.bluemind.backend.mail.api.ImapItemIdentifier;
import net.bluemind.backend.mail.api.MailboxItem;
import net.bluemind.backend.mail.api.flags.FlagUpdate;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.api.SubtreeLocation;
import net.bluemind.backend.mail.replica.indexing.IMailIndexService;
import net.bluemind.backend.mail.replica.service.internal.ImapMailboxRecordsService;
import net.bluemind.backend.mail.repository.IMailboxRecordStore;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.date.BmDateTime;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.Ack;
import net.bluemind.core.container.api.Count;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemChangelog;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.model.SortDescriptor;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.mailbox.api.IMailboxAclUids;

public class ImapMailboxRecordsServiceFactory extends AbstractMailboxRecordServiceFactory<IMailboxItems> {

	@Override
	public Class<IMailboxItems> factoryClass() {
		return IMailboxItems.class;
	}

	@Override
	protected IMailboxItems create(Container cont, BmContext context, String mailboxUniqueId,
			IMailboxRecordStore recordStore, ContainerStoreService<MailboxRecord> storeService,
			IMailIndexService indexService, IDbMessageBodies bodiesApi, Supplier<SubtreeLocation> location) {

		if (context.getSecurityContext().getSubject().equals(cont.owner)) {
			RBACManager ownerPerms = RBACManager.forContext(context).forContainer(cont);
			return new ImapMailboxRecordsService(cont, context, mailboxUniqueId, recordStore, storeService, ownerPerms,
					bodiesApi, location);
		}

		RBACManager mailboxPerms = RBACManager.forContext(context)
				.forContainer(IMailboxAclUids.uidForMailbox(cont.owner));
		if (mailboxPerms.can(Verb.Read.name())) {
			if (!mailboxPerms.can(Verb.Write.name())) {
				RBACManager folderPerms = RBACManager.forContext(context).forContainer(cont.uid);
				// folder has more rights than mailbox-acl for read-only mailbox
				if (folderPerms.can(Verb.Write.name())) {
					mailboxPerms = folderPerms;
				}
			}

			return new ImapMailboxRecordsService(cont, context, mailboxUniqueId, recordStore, storeService,
					mailboxPerms, bodiesApi, location);
		} else {
			mailboxPerms.check(Verb.Visible.name());
			RBACManager folderRbac = RBACManager.forContext(context).forContainer(cont.uid);
			if (folderRbac.can(Verb.Read.name())) {
				return new ImapMailboxRecordsService(cont, context, mailboxUniqueId, recordStore, storeService,
						folderRbac, bodiesApi, location);
			} else {
				return new VirtualFolder();
			}
		}

	}

	private static class VirtualFolder implements IMailboxItems {

		@Override
		public ItemChangelog itemChangelog(String itemUid, Long since) throws ServerFault {
			throw notSupported();
		}

		private ServerFault notSupported() {
			return new ServerFault("not on virtual folder");
		}

		@Override
		public ContainerChangeset<String> changeset(Long since) throws ServerFault {
			return ContainerChangeset.empty(since);
		}

		@Override
		public ContainerChangeset<Long> changesetById(Long since) throws ServerFault {
			return ContainerChangeset.empty(since);
		}

		@Override
		public ContainerChangeset<ItemVersion> filteredChangesetById(Long since, ItemFlagFilter filter)
				throws ServerFault {
			return ContainerChangeset.empty(since);
		}

		@Override
		public long getVersion() throws ServerFault {
			return 0;
		}

		@Override
		public void multipleDeleteById(List<Long> ids, Boolean bypassDeletedItems) throws ServerFault {
			throw notSupported();
		}

		@Override
		public ItemValue<MailboxItem> getCompleteById(long id) {
			return null;
		}

		@Override
		public List<ItemValue<MailboxItem>> multipleGetById(List<Long> ids) {
			return Collections.emptyList();
		}

		@Override
		public Count count(ItemFlagFilter filter) throws ServerFault {
			return Count.of(0);
		}

		@Override
		public List<Long> sortedIds(SortDescriptor sorted) throws ServerFault {
			return Collections.emptyList();
		}

		@Override
		public String uploadPart(Stream part) {
			throw notSupported();
		}

		@Override
		public List<Long> unreadItems() {
			return Collections.emptyList();
		}

		@Override
		public List<Long> recentItems(Date deliveredOrUpdatedAfter) {
			return Collections.emptyList();
		}

		@Override
		public void removePart(String partId) {
			throw notSupported();
		}

		@Override
		public ImapAck updateById(long id, MailboxItem value) {
			throw notSupported();
		}

		@Override
		public ImapAck createById(long id, MailboxItem value) {
			throw notSupported();
		}

		@Override
		public ImapItemIdentifier create(MailboxItem value) {
			throw notSupported();
		}

		@Override
		public void deleteById(long id) {
			throw notSupported();
		}

		@Override
		public Stream fetch(long imapUid, String address, String encoding, String mime, String charset,
				String filename) {
			throw notSupported();
		}

		@Override
		public ItemIdentifier unexpunge(long itemId) {
			throw notSupported();
		}

		@Override
		public List<ItemIdentifier> multipleUnexpungeById(List<Long> itemIds) {
			throw notSupported();
		}

		@Override
		public void expunge() {
		}

		@Override
		public Stream fetchComplete(long imapUid) {
			throw notSupported();
		}

		@Override
		public Stream fetchCompleteByGuid(String guid) {
			throw notSupported();
		}

		@Override
		public Ack addFlag(FlagUpdate flagUpdate) {
			throw notSupported();
		}

		@Override
		public Ack deleteFlag(FlagUpdate flagUpdate) {
			throw notSupported();
		}

		@Override
		public ItemValue<MailboxItem> getForUpdate(long id) {
			throw notSupported();
		}

		@Override
		public List<Long> listItemIdsAfter(BmDateTime before) {
			return Collections.emptyList();
		}

	}

}
