/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2019
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.sds;

import java.io.IOException;
import java.lang.ref.Reference;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.replica.api.TierMove;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.rest.BmContext;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.memory.pool.api.ChunkFiler;
import net.bluemind.memory.pool.api.ChunkFiler.NamedChunk;
import net.bluemind.network.topology.IServiceTopology;
import net.bluemind.network.topology.Topology;
import net.bluemind.sds.dto.DeleteRequest;
import net.bluemind.sds.dto.ExistRequest;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.MgetRequest;
import net.bluemind.sds.dto.MgetRequest.Transfer;
import net.bluemind.sds.dto.PutRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.dto.TierMoveRequest;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.sds.store.SdsException;
import net.bluemind.sds.store.loader.SdsStoreLoader;
import net.bluemind.sds.store.noop.NoopStoreFactory;
import net.bluemind.size.helper.MaxMessageSize;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.sysconf.helper.SysConfHelper;

public class MessageBodyObjectStore {

	private static final Logger logger = LoggerFactory.getLogger(MessageBodyObjectStore.class);
	private final BmContext ctx;
	private final ISdsSyncStore objectStore;
	private boolean singleNamespaceBody;
	private final String dataLocation;

	public MessageBodyObjectStore(BmContext ctx, DataLocation datalocation) {
		this.ctx = ctx;
		this.dataLocation = validLocation(datalocation.serverUid());
		SystemConf config = SysConfHelper.fromSharedMap();
		SdsStoreLoader loader = new SdsStoreLoader();
		this.singleNamespaceBody = Topology.get().singleNode() || !loader.archiveKind(config).isShardedByDatalocation();
		this.objectStore = loader.forSysconf(config, this.dataLocation).orElseGet(() -> {
			logger.warn("[{}] Returning noop store which might be wrong", datalocation);
			return new NoopStoreFactory().createSync(VertxPlatform.getVertx(), config, this.dataLocation);
		});
	}

	private String validLocation(String maybeServerUid) {
		IServiceTopology topo = Topology.get();
		return topo.nodes().stream().filter(iv -> iv.uid.equals(maybeServerUid)).findAny()
				.orElseGet(() -> topo.any("mail/imap")).uid;
	}

	public MessageBodyObjectStore(BmContext ctx, ISdsSyncStore objectStore, String datalocation) {
		this.ctx = ctx;
		this.objectStore = objectStore;
		this.dataLocation = validLocation(datalocation);
		if (logger.isDebugEnabled()) {
			logger.debug("Object store for {}: {}", this.ctx, objectStore);
		}
	}

	/**
	 * Only one namespace for all body guid's (eg. single S3 bucket for all)
	 *
	 * @return true if object store ignores datalocation
	 */
	public boolean isSingleNamespaceBody() {
		return singleNamespaceBody;
	}

	/**
	 * Returns a sub-collections containing only the GUIDs that our known in the
	 * object store
	 *
	 * @param bodyGuid
	 * @return
	 */
	public Set<String> exist(Set<String> guids) {
		logger.debug("Checking {} with {}", guids, objectStore);
		return guids.stream().filter(guid -> objectStore.exists(ExistRequest.of(guid)).exists)
				.collect(Collectors.toSet());

	}

	public NamedChunk open(String guid) {
		logger.debug("Open {} with {}", guid, objectStore);
		NamedChunk chunkAddr = null;
		try {
			chunkAddr = ChunkFiler.newNamedChunk(MaxMessageSize.get());
			SdsResponse resp = objectStore.download(GetRequest.of("", guid, chunkAddr.name()));
			if (resp.succeeded()) {
				return chunkAddr;
			} else {
				throw new ServerFault(resp.error.message);
			}
		} catch (Exception e) {
			if (chunkAddr != null) {
				ChunkFiler.byAddress(chunkAddr.name()).release();
			}
			throw new ServerFault(e);
		} finally {
			Reference.reachabilityFence(chunkAddr);
		}
	}

	public NamedChunk[] mopen(String[] guids) {
		if (guids.length == 1) {
			return new NamedChunk[] { open(guids[0]) };
		}
		logger.debug("Open {} with {}", guids, objectStore);
		List<Transfer> transfers = new ArrayList<>();
		ArrayList<NamedChunk> paths = new ArrayList<>();

		for (String guid : guids) {
			NamedChunk chunkAddr = null;
			try {
				chunkAddr = ChunkFiler.newNamedChunk(MaxMessageSize.get());
				paths.add(chunkAddr);
				transfers.add(Transfer.of(guid, chunkAddr.name()));
			} catch (IOException e) {
				paths.forEach(addr -> ChunkFiler.byAddress(addr.name()).release());
				throw new ServerFault(e);
			}
		}

		MgetRequest mgetReq = MgetRequest.of("unused", transfers);

		try {
			SdsResponse downloadResponse = objectStore.downloads(mgetReq);
			if (!downloadResponse.succeeded()) {
				throw new SdsException(downloadResponse);
			}
		} catch (Exception e) {
			paths.forEach(addr -> ChunkFiler.byAddress(addr.name()).release());
			if (e instanceof SdsException sdsE) {
				throw sdsE;
			}
			throw new ServerFault(e);
		}

		return paths.stream().toArray(NamedChunk[]::new);
	}

	public void delete(List<String> guids) {
		objectStore.delete(DeleteRequest.of(guids));
	}

	public void store(String uid, Date deliveryDate, String fileOrChunkAddress) {
		logger.debug("Store {} with {}", uid, objectStore);
		PutRequest pr = new PutRequest();
		pr.filename = fileOrChunkAddress;
		pr.guid = uid;
		pr.deliveryDate = deliveryDate;
		SdsResponse resp = objectStore.upload(pr);
		if (!resp.succeeded()) {
			throw new ServerFault("SDS backend refused upload of " + uid + ": " + resp.error);
		}
	}

	public List<String> tierMove(List<TierMove> tierMoves) {
		return objectStore.tierMove(new TierMoveRequest(tierMoves)).moved;
	}

	public String dataLocation() {
		return dataLocation;
	}

}
