/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal.repair;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.MaintenanceOperation;
import net.bluemind.directory.service.IDirEntryRepairSupport;
import net.bluemind.directory.service.RepairTaskMonitor;
import net.bluemind.mailbox.api.Mailbox;

public class BrokenTreeRepair implements IDirEntryRepairSupport {

	public static class RepairFactory implements IDirEntryRepairSupport.Factory {
		@Override
		public IDirEntryRepairSupport create(BmContext context) {
			return new BrokenTreeRepair(context);
		}
	}

	public static final MaintenanceOperation minboxOp = MaintenanceOperation.create(IMailReplicaUids.REPAIR_ORPHANS_OP,
			"Missing mail folders parents");

	private static class BrokenTreeMaintenance extends MailboxFoldersRepairOp {

		public BrokenTreeMaintenance(BmContext ctx) {
			super(ctx, minboxOp.identifier, null, IMailReplicaUids.REPAIR_SUBTREE_OP, 1);
		}

		@Override
		public void runOnFolders(boolean repair, RepairTaskMonitor monitor, String subTree, String domainUid,
				ItemValue<Mailbox> mbox, List<ItemValue<MailboxReplica>> folders) {
			IDbByContainerReplicatedMailboxes foldersApi = context.provider()
					.instance(IDbByContainerReplicatedMailboxes.class, subTree);

			monitor.begin(1, "Inspecting subtree for mailbox " + mbox.value.name + "@" + domainUid);
			Map<String, MailboxReplica> knownByUid = folders.stream().filter(f -> f.value != null)
					.collect(Collectors.toMap(iv -> iv.uid, iv -> iv.value));
			OpCounter count = new OpCounter(folders.size());

			folders.stream().filter(f -> f.value != null).forEach(folder -> {
				count.folder();
				String fullName = folder.value.fullName;
				String parentUid = folder.value.parentUid;

				// Root folder
				if (!fullName.contains("/")) {
					if (parentUid != null) {
						monitor.notify("{}: root folder {} -> '{}' has non null parent-uid: {}",
								repair ? "Attempting repair" : "Issue found", folder.uid, fullName, parentUid);
						if (repair) {
							updateFolderWithParentUid(monitor, foldersApi, folder, null, count);
						}
					}
					return;
				}

				// Sub folder
				String expectedParentPath = parentPath(fullName);
				if (parentUid != null) {
					MailboxReplica parent = knownByUid.get(parentUid);
					if (parent != null && parent.fullName.equals(expectedParentPath)) {
						// Folder hierarchy is OK
						return;
					}
				}

				// Repair sub folder parent
				monitor.notify("{}: sub folder {} -> '{}' has invalid parent: {}", //
						repair ? "Attempting repair" : "Issue found", folder.uid, fullName, parentUid);
				if (repair) {
					String actualParentUid = findParentUid(knownByUid, expectedParentPath)
							.orElseGet(() -> createParentFor(monitor, foldersApi, knownByUid, folder.value, count));

					updateFolderWithParentUid(monitor, foldersApi, folder, actualParentUid, count);
				}

			});

			monitor.end(true, count.report(), count.report());

		}

		public String parentPath(String fullName) {
			return fullName.substring(0, fullName.lastIndexOf("/"));
		}

		private String createParentFor(RepairTaskMonitor monitor, IDbByContainerReplicatedMailboxes foldersApi,
				Map<String, MailboxReplica> knownByUid, MailboxReplica childFolder, OpCounter count) {

			MailboxReplica newParent = new MailboxReplica();
			newParent.fullName = parentPath(childFolder.fullName);
			newParent.parentUid = !newParent.fullName.contains("/") ? null
					: findParentUid(knownByUid, parentPath(newParent.fullName))
							.orElseGet(() -> createParentFor(monitor, foldersApi, knownByUid, newParent, count));
			String newParentUid = childFolder.parentUid != null ? childFolder.parentUid : UUID.randomUUID().toString();

			try {
				foldersApi.create(newParentUid, newParent);
			} catch (RuntimeException e) {
				monitor.error("Error: could not create folder {} -> '{}' ({})", newParentUid, newParent.fullName,
						e.getMessage());
				count.error();
				return null;
			}

			knownByUid.put(newParentUid, newParent);
			count.create();
			monitor.notify("Repaired: created folder {} -> '{}'", newParentUid, newParent.fullName);
			return newParentUid;
		}

		public Optional<String> findParentUid(Map<String, MailboxReplica> knownByUid, String expectedParentPath) {
			return knownByUid.entrySet().stream() //
					.filter(e -> e.getValue().fullName.equals(expectedParentPath)) //
					.map(e -> e.getKey()) //
					.findAny();
		}

		private void updateFolderWithParentUid(RepairTaskMonitor monitor, IDbByContainerReplicatedMailboxes foldersApi,
				ItemValue<MailboxReplica> folder, String newParentUid, OpCounter count) {
			folder.value.parentUid = newParentUid;

			try {
				foldersApi.update(folder.uid, folder.value);
			} catch (RuntimeException e) {
				monitor.error("Error: could not update folder {} -> '{}' ({})", folder.uid, folder.value.fullName,
						e.getMessage());
				count.error();
			}

			count.update();
			monitor.notify("Repaired: updated folder {} -> '{}' with parent: {}", //
					folder.uid, folder.value.fullName, newParentUid);
		}
	}

	private static class OpCounter {
		private int totalFolders = 0;
		private int folders = 0;
		private int updates = 0;
		private int creates = 0;
		private int errors = 0;

		public OpCounter(int totalFolders) {
			this.totalFolders = totalFolders;
		}

		public void folder() {
			this.folders++;
		}

		public void update() {
			this.updates++;
		}

		public void create() {
			this.creates++;
		}

		public void error() {
			this.errors++;
		}

		public String report() {
			return "Repair on " + folders + "/" + totalFolders + " folders, performed " + updates + " updates and "
					+ creates + " creates." + (errors > 0 ? " (" + errors + " errors occured !)" : " (no error)");
		}
	}

	private final BmContext context;

	public BrokenTreeRepair(BmContext context) {
		this.context = context;
	}

	@Override
	public Set<MaintenanceOperation> availableOperations(Kind kind) {
		if (kind.hasMailbox()) {
			return Set.of(minboxOp);
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Set<InternalMaintenanceOperation> ops(Kind kind) {
		if (kind.hasMailbox()) {
			return Set.of(new BrokenTreeMaintenance(context));
		} else {
			return Collections.emptySet();
		}

	}
}
